-- Copyright 2024-2025 by Todd Hundersmarck (ThundR) 
-- All Rights Reserved

--[[

Unauthorized use and/or distribution of this work entitles
myself, the author, to unlimited free and unrestricted use,
access, and distribution of any works related to the unauthorized
user and/or distributor.

--]]

THBetterWinchAttachable = {}
local THBetterWinchAttachable_mt = Class(THBetterWinchAttachable)
THBetterWinchAttachable.CLASS_NAME = "THBetterWinchAttachable"
local debugFlagId = THUtils.createDebugFlagId(THBetterWinchAttachable.CLASS_NAME)
THBetterWinchAttachable.debugFlagId = debugFlagId
function THBetterWinchAttachable.new(target, customMt)
    customMt = customMt or THBetterWinchAttachable_mt
    if THUtils.argIsValid(THUtils.getIsType(target, Object), "target", target)
        and THUtils.argIsValid(type(customMt) == THValueType.TABLE, "customMt", customMt)
    then
        local sizeX, sizeY, sizeZ = g_thBetterWinch:getObjectSize(target)
        if sizeX == nil or sizeY == nil or sizeZ == nil then
            return
        end
        local self = setmetatable({}, customMt)
        self.index = 0
        self.target = target
        if type(target.getName) == THValueType.FUNCTION then
            self.targetName = target:getName()
        end
        self.targetName = self.targetName or "Attachable Object"
        self.size = { sizeX, sizeY, sizeZ }
        self.components = {
            byId = {},
            byIndex = {},
        }
        self.meshNodes = {
            byId = {},
            byIndex = {}
        }
        self.attachedWinches = {}
        self.componentAttachPoints = {}
        self.isDirty = false
        self.isLoadFinished = false
        return self
    end
end
function THBetterWinchAttachable.load(self)
    if self.isLoadFinished then
        return true
    end
    local success = true
    if type(self.target.components) == THValueType.TABLE
        and type(self.target.components[1]) == THValueType.TABLE
    then
        for compIndex = 1, #self.target.components do
            local componentInfo = self.target.components[compIndex]
            if self:addComponent(componentInfo.node) == nil then
                success = false
                break
            end
        end
    elseif type(self.target.nodeId) == THValueType.NUMBER and self.target.nodeId > 0 then
        if self:addComponent(self.target.nodeId) == nil then
            success = false
        end
    elseif type(self.target.rootNode) == THValueType.NUMBER and self.target.rootNode > 0 then
        if self:addComponent(self.target.rootNode) == nil then
            success = false
        end
    end
    if success and #self.components.byIndex <= 0 then
        success = false
    end
    if success then
        local attachPoints = g_thBetterWinch:getObjectAttachPoints(self.target)
        if attachPoints ~= nil then
            for _, attachPointData in ipairs(attachPoints) do
                self:addMeshNode(attachPointData.node)
            end
        end
        if self.target.getSupportsTensionBelts ~= nil
            and self.target:getSupportsTensionBelts()
            and self.target.getMeshNodes ~= nil
        then
            local tensionBeltNodes = self.target:getMeshNodes()
            if tensionBeltNodes ~= nil then
                for _, tensionBeltNode in ipairs(tensionBeltNodes) do
                    self:addMeshNode(tensionBeltNode)
                end
            end
        end
    end
    if success and #self.meshNodes.byIndex <= 0 then
        success = false
    end
    self.isLoadFinished = true
    return success
end
function THBetterWinchAttachable.delete(self)
    local numComponents = #self.components.byIndex
    local numMeshNodes = #self.meshNodes.byIndex
    if numMeshNodes > 0 then
        for meshIndex = numMeshNodes, 1, -1 do
            local meshData = self.meshNodes.byIndex[meshIndex]
            g_thBetterWinch:removeObjectInjectedAttachPoint(self.target, meshData.node)
        end
        THUtils.clearTable(self.meshNodes.byId)
        THUtils.clearTable(self.meshNodes.byIndex)
    end
    if numComponents > 0 then
        for componentIndex = numComponents, 1, -1 do
            local componentData = self.components.byIndex[componentIndex]
            g_thBetterWinch:removeObjectInjectedAttachPoint(self.target, componentData.node)
        end
        THUtils.clearTable(self.components.byId)
        THUtils.clearTable(self.components.byIndex)
    end
    self.lastAttachedWinch = nil
    self.lastAttachedComponentIndex = nil
    self.lastAttachedMeshNodeIndex = nil
    self.lastTargetedComponentIndex = nil
    self.lastTargetedMeshNodeIndex = nil
    return true
end
function THBetterWinchAttachable.addComponent(self, nodeId, verbose)
    if THUtils.argIsValid(not verbose or verbose == true, "verbose", verbose) then
        if g_thBetterWinch:validateAttachableObjectNodeId(nodeId, verbose)
            and self.components.byId[nodeId] == nil
        then
            local componentData = {
                index = #self.components.byIndex + 1,
                parent = self,
                node = nodeId,
                nodeName = getName(nodeId)
            }
            local sizeX = getUserAttribute(nodeId, "thWidth")
            local sizeY = getUserAttribute(nodeId, "thHeight")
            local sizeZ = getUserAttribute(nodeId, "thLength")
            if type(sizeX) ~= THValueType.NUMBER or sizeX <= 0 then
                sizeX = self.size[1]
            end
            if type(sizeY) ~= THValueType.NUMBER or sizeY <= 0 then
                sizeY = self.size[2]
            end
            if type(sizeZ) ~= THValueType.NUMBER or sizeZ <= 0 then
                sizeZ = self.size[3]
            end
            componentData.size = { sizeX, sizeY, sizeZ }
            self.components.byId[componentData.node] = componentData
            self.components.byIndex[componentData.index] = componentData
            return componentData
        end
    end
end
function THBetterWinchAttachable.addMeshNode(self, nodeId, verbose)
    if THUtils.argIsValid(not verbose or verbose == true, "verbose", verbose) then
        if type(nodeId) ~= THValueType.NUMBER or nodeId <= 0 then
            if verbose then
                THUtils.printDebugError(debugFlagId, true, THMessage.ARGUMENT_INVALID, "nodeId", nodeId)
            end
        elseif not THUtils.getNodeExists(nodeId) then
            if verbose then
                THUtils.printDebugError(debugFlagId, nil, "Mesh node %q does not exist", nodeId)
            end
        else
            local nodeName = getName(nodeId)
            if not getHasClassId(nodeId, ClassIds.SHAPE) then
                if verbose then
                    THUtils.printDebugError(debugFlagId, nil, "Mesh node %q [%s] is not a valid shape", nodeName, nodeId)
                end
            elseif not getShapeIsCPUMesh(nodeId) then
                if verbose then
                    THUtils.printDebugError(debugFlagId, nil, "Mesh node %q [%s] must have cpu mesh flag enabled", nodeName, nodeId)
                end
            elseif self.meshNodes.byId[nodeId] == nil then
                local meshData = {
                    index = #self.meshNodes.byIndex + 1,
                    node = nodeId,
                    nodeName = nodeName,
                    parent = self,
                    upAxis = THAxis.Y,
                    lastAxisRadius = {},
                    rigidBodyType = RigidBodyType.NONE,
                    canTarget = false
                }
                if getHasCollision(nodeId) then
                    meshData.rigidBodyType = getRigidBodyType(meshData.node)
                    if CollisionFlag.getHasGroupFlagSet(nodeId, THBetterWinch.COLLISION_MASK)
                        and meshData.rigidBodyType ~= nil and meshData.rigidBodyType ~= RigidBodyType.NONE
                    then
                        meshData.canTarget = true
                    end
                end
                local componentData = self:getComponent(nodeId, true)
                local isUserAttachPoint = g_thBetterWinch:getAttachPointNode(nodeId) ~= nil
                local sizeX = getUserAttribute(nodeId, "thWidth")
                local sizeY = getUserAttribute(nodeId, "thHeight")
                local sizeZ = getUserAttribute(nodeId, "thLength")
                if type(sizeX) ~= THValueType.NUMBER or sizeX <= 0 then
                    if isUserAttachPoint
                        or (meshData.canTarget and componentData == nil)
                    then
                        sizeX = THBetterWinch.DEFAULT_ATTACH_RADIUS * 2
                    elseif componentData ~= nil then
                        sizeX = componentData.size[1]
                    else
                        sizeX = self.size[1]
                    end
                end
                if type(sizeY) ~= THValueType.NUMBER or sizeY <= 0 then
                    if isUserAttachPoint
                        or (meshData.canTarget and componentData == nil)
                    then
                        sizeY = THBetterWinch.DEFAULT_ATTACH_RADIUS * 2
                    elseif componentData ~= nil then
                        sizeY = componentData.size[2]
                    else
                        sizeY = self.size[2]
                    end
                end
                if type(sizeZ) ~= THValueType.NUMBER or sizeZ <= 0 then
                    if isUserAttachPoint
                        or (meshData.canTarget and componentData == nil)
                    then
                        sizeZ = THBetterWinch.DEFAULT_ATTACH_RADIUS * 2
                    elseif componentData ~= nil then
                        sizeZ = componentData.size[3]
                    else
                        sizeZ = self.size[3]
                    end
                end
                meshData.size = { sizeX, sizeY, sizeZ }
                meshData.lastAxisRadius[THAxis.X] = math.max(sizeY, sizeZ) * 0.5
                meshData.lastAxisRadius[THAxis.Y] = math.max(sizeX, sizeZ) * 0.5
                meshData.lastAxisRadius[THAxis.Z] = math.max(sizeX, sizeY) * 0.5
                meshData.radius = meshData.lastAxisRadius[meshData.upAxis] + 0.01
                meshData.hitRadius = (math.max(sizeX, sizeY, sizeZ) * 0.5) + 0.01
                self.meshNodes.byId[meshData.node] = meshData
                self.meshNodes.byIndex[meshData.index] = meshData
                return meshData
            end
        end
    end
end
function THBetterWinchAttachable.getIsEnabled(self)
    if self.target ~= nil and self.target.isRegistered and not self.target.isDeleted then
        return true
    end
    return false
end
function THBetterWinchAttachable.getComponent(self, nodeId, force, verbose)
    if THUtils.argIsValid(not force or force == true, "force", force)
        and THUtils.argIsValid(not verbose or verbose == true, "verbose", verbose)
    then
        if nodeId ~= nil then
            local componentData = self.components.byId[nodeId]
            if componentData ~= nil then
                if THUtils.getNodeExists(componentData.node) or force then
                    return componentData
                end
                if verbose then
                    THUtils.printDebugError(debugFlagId, nil, "Component %q [%s] does not exist", componentData.nodeName, componentData.node)
                end
                return
            end
        end
        if verbose then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "nodeId", nodeId)
        end
    end
end
function THBetterWinchAttachable.getComponentByIndex(self, index, force, verbose)
    if THUtils.argIsValid(not force or force == true, "force", force)
        and THUtils.argIsValid(not verbose or verbose == true, "verbose", verbose)
    then
        if index ~= nil then
            local componentData = self.components.byIndex[index]
            if componentData ~= nil then
                if THUtils.getNodeExists(componentData.node) or force then
                    return componentData
                end
                if verbose then
                    THUtils.printDebugError(debugFlagId, nil, "Component %q [%s] does not exist", componentData.nodeName, componentData.node)
                end
                return
            end
        end
        if verbose then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "index", index)
        end
    end
end
function THBetterWinchAttachable.getComponents(self)
    return self.components.byIndex, #self.components.byIndex
end
function THBetterWinchAttachable.getMeshNode(self, nodeId, force, verbose)
    if THUtils.argIsValid(not force or force == true, "force", force)
        and THUtils.argIsValid(not verbose or verbose == true, "verbose", verbose)
    then
        if nodeId ~= nil then
            local meshData = self.meshNodes.byId[nodeId]
            if meshData ~= nil then
                if THUtils.getNodeExists(meshData.node) or force then
                    return meshData
                end
                if verbose then
                    THUtils.printDebugError(debugFlagId, nil, "Mesh node %q [%s] does not exist", meshData.nodeName, meshData.node)
                end
                return
            end
        end
        if verbose then
            THUtils.printDebugError(debugFlagId, true, THMessage.ARGUMENT_INVALID, "nodeId", nodeId)
        end
    end
end
function THBetterWinchAttachable.getMeshNodeByIndex(self, index, force, verbose)
    if THUtils.argIsValid(not force or force == true, "force", force)
        and THUtils.argIsValid(not verbose or verbose == true, "verbose", verbose)
    then
        if index ~= nil then
            local meshData = self.meshNodes.byIndex[index]
            if meshData ~= nil then
                if THUtils.getNodeExists(meshData.node) or force then
                    return meshData
                end
                if verbose then
                    THUtils.printDebugError(debugFlagId, nil, "Mesh node %q [%s] does not exist", meshData.nodeName, meshData.node)
                end
                return
            end
        end
        if verbose then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "index", index)
        end
    end
end
function THBetterWinchAttachable.getMeshNodes(self)
    return self.meshNodes.byIndex, #self.meshNodes.byIndex
end
function THBetterWinchAttachable.getLastTargetData(self, force)
    force = THUtils.validateArg(not force or force == true, "force", force, false)
    local componentData = nil
    local meshData = nil
    if self.lastTargetedComponentIndex ~= nil then
        componentData = self:getComponentByIndex(self.lastTargetedComponentIndex, force)
    end
    if self.lastTargetedMeshNodeIndex ~= nil then
        meshData = self:getMeshNodeByIndex(self.lastTargetedMeshNodeIndex, force)
    end
    return componentData, meshData
end
function THBetterWinchAttachable.getClosestMeshNode(self, x, y, z, maxDistance, force, refNode)
    if THUtils.argIsValid(maxDistance == nil or (type(maxDistance) == THValueType.NUMBER and maxDistance > 0), "maxDistance", maxDistance)
        and THUtils.argIsValid(type(x) == THValueType.NUMBER, "x", x)
        and THUtils.argIsValid(type(y) == THValueType.NUMBER, "y", y)
        and THUtils.argIsValid(type(z) == THValueType.NUMBER, "z", z)
        and THUtils.argIsValid(not force or force == true, "force", force)
        and THUtils.argIsValid(refNode == nil or (type(refNode) == THValueType.NUMBER and refNode > 0), "refNode", refNode)
    then
        local foundMeshData = nil
        local minMeshDistance = maxDistance
        local numMeshNodes = #self.meshNodes.byIndex
        if numMeshNodes > 0 then
            for meshIndex = 1, numMeshNodes do
                local meshData = self.meshNodes.byIndex[meshIndex]
                if THUtils.getNodeExists(meshData.node)
                    and (foundMeshData == nil or meshData.node ~= foundMeshData.node)
                then
                    local lx, ly, lz = worldToLocal(meshData.node, x, y, z)
                    local meshDistance = THUtils.getVector3Distance(lx, ly, lz)
                    if meshDistance ~= nil
                        and (minMeshDistance == nil or meshDistance < minMeshDistance)
                    then
                        local isMeshNodeValid = false
                        if force or meshDistance <= meshData.hitRadius then
                            isMeshNodeValid = true
                        elseif refNode ~= nil then
                            if meshData.node == refNode
                                or (foundMeshData == nil and not meshData.canTarget)
                            then
                                isMeshNodeValid = true
                            end
                        end
                        if isMeshNodeValid then
                            foundMeshData = meshData
                            minMeshDistance = meshDistance
                        end
                    end
                end
            end
        end
        if foundMeshData ~= nil then
            return foundMeshData, minMeshDistance
        end
    end
end
function THBetterWinchAttachable.getIsDirty(self)
    return self.isDirty == true
end
function THBetterWinchAttachable.setIsDirty(self, isDirty)
    isDirty = THUtils.getNoNil(isDirty, true)
    if THUtils.argIsValid(not isDirty or isDirty == true, "isDirty", isDirty) then
        if self.isDirty ~= isDirty then
            self.isDirty = isDirty == true
        end
        if self:getIsDirty() then
            if self.target.raiseActive ~= nil then
                self.target:raiseActive()
            end
        end
    end
end
function THBetterWinchAttachable.updateWinchAttachData(self, targetWinch)
    local specTable = THVSpec_Winch.getSpecTable(targetWinch)
    if THUtils.argIsValid(specTable ~= nil, "targetWinch", targetWinch) then
        local winchSpec = targetWinch.spec_winch
        if targetWinch == self.lastAttachedWinch then
            self.lastAttachedWinch = nil
            self.lastAttachedComponentIndex = nil
            self.lastAttachedMeshNodeIndex = nil
        end
        for _, componentAttachPoints in pairs(self.componentAttachPoints) do
            local numAttachPoints = #componentAttachPoints
            if numAttachPoints > 0 then
                for attachPointIndex = numAttachPoints, 1, -1 do
                    local attachedComponentData = componentAttachPoints[attachPointIndex]
                    if attachedComponentData.winch == targetWinch then
                        componentAttachPoints[attachPointIndex] = nil
                    end
                end
            end
        end
        if winchSpec.ropes ~= nil and next(winchSpec.ropes) ~= nil then
            for ropeIndex, ropeDesc in pairs(winchSpec.ropes) do
                if ropeDesc.attachedTrees ~= nil and next(ropeDesc.attachedTrees) ~= nil then
                    for _, ropeAttachDesc in pairs(ropeDesc.attachedTrees) do
                        local componentData = self:getComponent(ropeAttachDesc.treeId)
                        if componentData ~= nil then
                            local attachedComponentData = nil
                            local attachedRopes = self.attachedWinches[targetWinch]
                            if attachedRopes ~= nil then
                                local attachedComponents = attachedRopes[ropeIndex]
                                if attachedComponents ~= nil then
                                    attachedComponentData = attachedComponents[componentData.index]
                                end
                            end
                            if attachedComponentData == nil then
                                local meshData = self:getMeshNodeByIndex(self.lastTargetedMeshNodeIndex)
                                if meshData == nil then
                                    local wx, wy, wz = getWorldTranslation(componentData.node)
                                    meshData = self:getClosestMeshNode(wx, wy, wz)
                                end
                                if meshData == nil then
                                    THUtils.errorMsg(true, "Cannot find suitable mesh node for component %q [%s]", componentData.nodeName, componentData.node)
                                else
                                    attachedComponentData = {
                                        winch = targetWinch,
                                        ropeIndex = ropeIndex,
                                        ropeDesc = ropeDesc,
                                        ropeAttachDesc = ropeAttachDesc,
                                        component = componentData,
                                        meshNode = meshData
                                    }
                                    self.lastAttachedWinch = targetWinch
                                    self.lastAttachedComponentIndex = componentData.index
                                    self.lastAttachedMeshNodeIndex = meshData.index
                                end
                            end
                            if attachedComponentData ~= nil then
                                local componentAttachPoints = self.componentAttachPoints[attachedComponentData.component.index]
                                if componentAttachPoints == nil then
                                    componentAttachPoints = {}
                                    self.componentAttachPoints[attachedComponentData.component.index] = componentAttachPoints
                                end
                                table.insert(componentAttachPoints, 1, attachedComponentData)
                            end
                        end
                    end
                end
            end
        end
        local attachedRopes = self.attachedWinches[targetWinch]
        if attachedRopes == nil then
            attachedRopes = {}
            self.attachedWinches[targetWinch] = attachedRopes
        else
            THUtils.clearTable(attachedRopes)
        end
        for _, componentAttachPoints in pairs(self.componentAttachPoints) do
            for _, attachedComponentData in pairs(componentAttachPoints) do
                if attachedComponentData.winch == targetWinch then
                    local attachedComponents = attachedRopes[attachedComponentData.ropeIndex]
                    if attachedComponents == nil then
                        attachedComponents = {}
                        attachedRopes[attachedComponentData.ropeIndex] = attachedComponents
                    end
                    attachedComponents[attachedComponentData.component.index] = attachedComponentData
                end
            end
        end
        self:setIsDirty()
    end
end
function THBetterWinchAttachable.getIsAttached(self, targetWinch, ropeIndex, componentIndex, meshIndex)
    local componentData = nil
    local meshData = nil
    if componentIndex ~= nil then
        componentData = self:getComponentByIndex(componentIndex)
        if componentData == nil then
            return false
        end
    end
    if meshIndex ~= nil then
        meshData = self:getMeshNodeByIndex(meshIndex)
        if meshData == nil then
            return false
        end
    end
    if targetWinch == nil then
        for _, attachedRopes in pairs(self.attachedWinches) do
            for otherRopeIndex, attachedComponentData in pairs(attachedRopes) do
                if (ropeIndex == nil or ropeIndex == otherRopeIndex)
                    and (componentData == nil or attachedComponentData.component.index == componentData.index)
                    and (meshData == nil or attachedComponentData.meshNode.index == meshData.index)
                then
                    return true
                end
            end
        end
    else
        local specTable = THVSpec_Winch.getSpecTable(targetWinch)
        if THUtils.argIsValid(specTable ~= nil, "targetWinch", targetWinch) then
            local attachedRopes = self.attachedWinches[targetWinch]
            if attachedRopes == nil then
                return false
            end
            if ropeIndex == nil then
                for _, attachedComponentData in pairs(attachedRopes) do
                    if (componentData == nil or attachedComponentData.component.index == componentData.index)
                        and (meshData == nil or attachedComponentData.meshNode.index == meshData.index)
                    then
                        return true
                    end
                end
            else
                local attachedComponentData = attachedRopes[ropeIndex]
                if attachedComponentData ~= nil
                    and (componentData == nil or attachedComponentData.component.index == componentData.index)
                    and (meshData == nil or attachedComponentData.meshNode.index == meshData.index)
                then
                    return true
                end
            end
        end
    end
    return false
end
function THBetterWinchAttachable.getActiveWinch(self, componentIndex)
    local winchVehicle, winchRootVehicle = nil, nil
    if self.lastAttachedWinch ~= nil and not self.lastAttachedWinch.isDeleted then
        winchVehicle = self.lastAttachedWinch
    else
        local componentData = nil
        if componentIndex ~= nil then
            componentData = self:getComponentByIndex(componentIndex)
            if componentData == nil then
                THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "componentIndex", componentIndex)
                return
            end
        elseif self.lastAttachedComponentIndex ~= nil then
            componentData = self:getComponentByIndex(self.lastAttachedComponentIndex)
        end
        if componentData == nil then
            for _, componentAttachPoints in pairs(self.componentAttachPoints) do
                local attachedComponentData = componentAttachPoints[1]
                if attachedComponentData ~= nil then
                    winchVehicle = attachedComponentData.winch
                    break
                end
            end
        else
            local componentAttachPoints = self.componentAttachPoints[componentData.index]
            if componentAttachPoints ~= nil then
                local attachedComponentData = componentAttachPoints[1]
                if attachedComponentData ~= nil then
                    winchVehicle = attachedComponentData.winch
                end
            end
        end
    end
    if winchVehicle ~= nil then
        winchRootVehicle = winchVehicle
        if type(winchVehicle.getRootVehicle) ~= nil then
            winchRootVehicle = winchVehicle:getRootVehicle() or winchVehicle
        end
        return winchVehicle, winchRootVehicle
    end
end