-- Copyright 2024-2025 by Todd Hundersmarck (ThundR) 
-- All Rights Reserved

--[[

Unauthorized use and/or distribution of this work entitles
myself, the author, to unlimited free and unrestricted use,
access, and distribution of any works related to the unauthorized
user and/or distributor.

--]]

local g_thModName = g_thBetterWinch.coreData.mod.name
THVSpec_Winch = {}
THVSpec_Winch.CLASS_NAME = "THVSpec_Winch"
THVSpec_Winch.SPEC_NAME = "thWinch"
THVSpec_Winch.SPEC_KEY = "spec_" .. g_thModName .. "." .. THVSpec_Winch.SPEC_NAME
THVSpec_Winch.ACTION_TEXT = {
    ADJUST_ATTACH_RADIUS = g_i18n:getText("thAction_adjustAttachRadius"),
    ROTATE_ATTACH_LOOP = g_i18n:getText("thAction_rotateAttachLoop")
}
local debugFlagId = THUtils.createDebugFlagId(THVSpec_Winch.CLASS_NAME)
THVSpec_Winch.debugFlagId = debugFlagId
THWinchAttachTreeActivatable = {}
local THWinchAttachTreeActivatable_mt = Class(THWinchAttachTreeActivatable)
function THVSpec_Winch.getSpecTable(self)
    return THUtils.call(function()
        if self ~= nil then
            return self[THVSpec_Winch.SPEC_KEY]
        end
    end)
end
local getSpecTable = THVSpec_Winch.getSpecTable
local function initScript()
    THUtils.overwriteFunction(g_thGlobalEnv, "raycastClosestAsync", false, false, nil, THVSpec_Winch.gOverwrite_raycastClosestAsync)
    THUtils.appendFunction("Winch", "registerXMLPaths", false, false, nil, THVSpec_Winch.gAppend_registerXMLPaths)
    THUtils.overwriteFunction("Winch", "saveToXMLFile", false, true, nil, THVSpec_Winch.gOverwrite_saveToXMLFile)
    THUtils.appendFunction("WinchAttachTreeActivatable", "registerCustomInput", false, false, nil, THWinchAttachTreeActivatable.gAppend_registerCustomInput)
    THUtils.appendFunction("WinchAttachTreeActivatable", "removeCustomInput", false, false, nil, THWinchAttachTreeActivatable.gAppend_removeCustomInput)
    THUtils.appendFunction("WinchAttachTreeActivatable", "update", false, false, nil, THWinchAttachTreeActivatable.gAppend_update)
    THUtils.overwriteFunction("WinchAttachTreeActivatable", "new", false, false, nil, THWinchAttachTreeActivatable.gOverwrite_new)
end
function THVSpec_Winch.prerequisitesPresent(specializations)
    if SpecializationUtil.hasSpecialization(Winch, specializations) then
        return true
    end
    return false
end
function THVSpec_Winch.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onPreLoad", THVSpec_Winch)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", THVSpec_Winch)
    SpecializationUtil.registerEventListener(vehicleType, "onLoadFinished", THVSpec_Winch)
    SpecializationUtil.registerEventListener(vehicleType, "onPostUpdate", THVSpec_Winch)
    SpecializationUtil.registerEventListener(vehicleType, "onReadStream", THVSpec_Winch)
    SpecializationUtil.registerEventListener(vehicleType, "onWriteStream", THVSpec_Winch)
end
function THVSpec_Winch.registerOverwrittenFunctions(vehicleType)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "attachTreeToWinch", THVSpec_Winch.overwrite_attachTreeToWinch)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "detachTreeFromWinch", THVSpec_Winch.overwrite_detachTreeFromWinch)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "onWinchTreeRaycastCallback", THVSpec_Winch.overwrite_onWinchTreeRaycastCallback)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "onWinchTreeShapeMounted", THVSpec_Winch.overwrite_onWinchTreeShapeMounted)
end
function THVSpec_Winch.onPreLoad(self, savegame)
    THUtils.pcall(function()
        local specTable = getSpecTable(self)
        local winchSpec = self.spec_winch
        if specTable ~= nil and winchSpec ~= nil then
            specTable.vehicle = self
            specTable.objectOnLoadData = {}
            specTable.winchRopeCache = {}
            specTable.attachedObjectsCache = {}
            specTable.isAttachPointRaycastMode = false
            specTable.onWinchAttachPointRaycastCallback = function(...)
                return THUtils.call(THVSpec_Winch.onWinchAttachPointRaycastCallback, ...)
            end
        end
    end)
end
function THVSpec_Winch.onLoad(self, savegame)
    THUtils.pcall(function()
        local specTable = getSpecTable(self)
        local winchSpec = self.spec_winch
        if specTable ~= nil and winchSpec ~= nil then
            if winchSpec.isAttachable then
                if SpecializationUtil.hasSpecialization(Enterable, self.specializations) then
                    winchSpec.isAttachable = false
                end
            end
            if winchSpec.ropes ~= nil and #winchSpec.ropes > 0
                and MessageType.PLAYER_PRE_TELEPORT ~= nil
                and type(self.onPlayerPreTeleport) == THValueType.FUNCTION
            then
                g_messageCenter:unsubscribe(MessageType.PLAYER_PRE_TELEPORT, self)
                g_messageCenter:subscribe(MessageType.PLAYER_PRE_TELEPORT, self.onPlayerPreTeleport, self)
            end
            local winchTexts = winchSpec.texts
            winchTexts.attachAnotherTree = g_i18n:getText("thBetterWinch_attachAnotherObject")
            winchTexts.attachTree = g_i18n:getText("thBetterWinch_attachObject")
            winchTexts.detachTree = g_i18n:getText("thBetterWinch_detachObject")
            winchTexts.warningTooHeavy = g_i18n:getText("thBetterWinch_objectTooHeavy")
            winchTexts.warningMaxNumTreesReached = g_i18n:getText("thBetterWinch_maxObjectsReached")
        end
    end)
end
function THVSpec_Winch.onLoadFinished(self, savegame)
    THUtils.pcall(function()
        local mission = g_thBetterWinch.coreData.mission
        local specTable = getSpecTable(self)
        local winchSpec = self.spec_winch
        if mission ~= nil and specTable ~= nil and winchSpec ~= nil
            and savegame ~= nil and savegame.xmlFile ~= nil
            and not savegame.resetVehicles
        then
            local xmlFile = savegame.xmlFile
            local xmlSubKey = savegame.key .. "." .. g_thBetterWinch.coreData.xmlKey
            xmlFile:iterate(xmlSubKey .. ".rope", function(_, pRopeKey)
                local ropeIndex = xmlFile:getInt(pRopeKey .. "#index")
                if ropeIndex ~= nil and ropeIndex >= 1 then
                    xmlFile:iterate(pRopeKey .. ".attachedObject", function(_, pObjectKey)
                        local objectUniqueId = xmlFile:getValue(pObjectKey .. "#uniqueId")
                        local componentIndex = xmlFile:getValue(pObjectKey .. "#component")
                        local translation = xmlFile:getValue(pObjectKey .. "#translation", nil, true)
                        local positionData = ForestryPhysicsRope.loadPositionDataFromSavegame(xmlFile, pObjectKey .. ".physicsRope")
                        if componentIndex ~= nil and positionData ~= nil
                            and objectUniqueId ~= nil and objectUniqueId ~= ""
                            and translation ~= nil and #translation == 3
                        then
                            local meshKey = pObjectKey .. ".meshNode"
                            local meshIndex = xmlFile:getValue(meshKey .. "#index")
                            local radius = xmlFile:getValue(meshKey .. "#radius")
                            local upAxis = xmlFile:getValue(meshKey .. "#upAxis")
                            if meshIndex ~= nil then
                                local objectOnLoadData = {
                                    uniqueId = objectUniqueId,
                                    ropeIndex = ropeIndex,
                                    componentIndex = componentIndex,
                                    meshIndex = meshIndex,
                                    radius = radius,
                                    upAxis = upAxis,
                                    positionData = positionData,
                                    translation = translation
                                }
                                table.insert(specTable.objectOnLoadData, objectOnLoadData)
                            end
                        end
                    end)
                end
            end)
        end
    end)
end
function THVSpec_Winch.onPostUpdate(self, dt)
    THUtils.pcall(function()
        local mission = g_thBetterWinch.coreData.mission
        local specTable = getSpecTable(self)
        local winchSpec = self.spec_winch
        if mission ~= nil and specTable ~= nil and winchSpec ~= nil then
            local numOnLoadObjects = #specTable.objectOnLoadData
            if numOnLoadObjects > 0 then
                for onLoadIndex = numOnLoadObjects, 1, -1 do
                    local objectOnLoadData = specTable.objectOnLoadData[onLoadIndex]
                    local object = mission:getObjectByUniqueId(objectOnLoadData.uniqueId)
                    if object ~= nil then
                        local objectData = g_thBetterWinch:registerAttachableObject(object)
                        if objectData ~= nil then
                            local componentData = objectData:getComponentByIndex(objectOnLoadData.componentIndex)
                            local meshData = objectData:getMeshNodeByIndex(objectOnLoadData.meshIndex)
                            if componentData ~= nil and meshData ~= nil then
                                objectData.lastTargetedComponentIndex = componentData.index
                                objectData.lastTargetedMeshNodeIndex = meshData.index
                                if objectOnLoadData.radius ~= nil then
                                    meshData.radius = objectOnLoadData.radius
                                end
                                if objectOnLoadData.upAxis ~= nil then
                                    meshData.upAxis = objectOnLoadData.upAxis
                                end
                                local wx, wy, wz = THUtils.unpack(objectOnLoadData.translation)
                                self:attachTreeToWinch(componentData.node, wx, wy, wz, objectOnLoadData.ropeIndex, nil, true)
                            end
                        end
                    end
                    specTable.objectOnLoadData[onLoadIndex] = nil
                end
            end
        end
    end)
end
function THVSpec_Winch.onReadStream(self, streamId, connection)
    local specTable = getSpecTable(self)
    if specTable ~= nil and connection:getIsServer() then
        THUtils.pcall(function()
            THUtils.clearTable(specTable.objectOnLoadData)
        end)
        local numOnLoadValues = streamReadUIntN(streamId, 16)
        if numOnLoadValues > 0 then
            for _ = 1, numOnLoadValues do
                local uniqueId = streamReadString(streamId)
                local ropeIndex = streamReadUIntN(streamId, 8)
                local componentIndex = streamReadUIntN(streamId, 16)
                if streamReadBool(streamId) then
                    local meshIndex = streamReadUIntN(streamId, 16)
                    local radius = streamReadFloat32(streamId)
                    local upAxis = streamReadUIntN(streamId, 4)
                    local wx = streamReadFloat32(streamId)
                    local wy = streamReadFloat32(streamId)
                    local wz = streamReadFloat32(streamId)
                    THUtils.pcall(function()
                        local translation = THUtils.pack(wx, wy, wz)
                        local objectOnLoadData = {
                            uniqueId = uniqueId,
                            ropeIndex = ropeIndex,
                            componentIndex = componentIndex,
                            meshIndex = meshIndex,
                            translation = translation
                        }
                        if radius > 0 then
                            objectOnLoadData.radius = radius
                        end
                        if upAxis > 0 then
                            objectOnLoadData.upAxis = upAxis
                        end
                        table.insert(specTable.objectOnLoadData, objectOnLoadData)
                    end)
                end
            end
        end
    end
end
function THVSpec_Winch.onWriteStream(self, streamId, connection)
    local specTable = getSpecTable(self)
    if specTable ~= nil and not connection:getIsServer() then
        local numOnLoadValues = #specTable.objectOnLoadData
        streamWriteUIntN(streamId, numOnLoadValues, 16)
        if numOnLoadValues > 0 then
            for onLoadIndex = 1, numOnLoadValues do
                local objectOnLoadData = specTable.objectOnLoadData[onLoadIndex]
                streamWriteString(streamId, objectOnLoadData.uniqueId)
                streamWriteUIntN(streamId, objectOnLoadData.ropeIndex, 8)
                streamWriteUIntN(streamId, objectOnLoadData.componentIndex, 16)
                if streamWriteBool(streamId, objectOnLoadData.meshIndex ~= nil) then
                    streamWriteUIntN(streamId, objectOnLoadData.meshIndex, 16)
                    streamWriteFloat32(streamId, objectOnLoadData.radius or 0)
                    streamWriteUIntN(streamId, objectOnLoadData.upAxis or 0, 4)
                    local wx, wy, wz = THUtils.unpack(objectOnLoadData.translation)
                    streamWriteFloat32(streamId, wx)
                    streamWriteFloat32(streamId, wy)
                    streamWriteFloat32(streamId, wz)
                end
            end
        end
    end
end
function THVSpec_Winch.adjustWinchAttachRadius(self, adjustValue)
    local specTable = getSpecTable(self)
    local winchSpec = self.spec_winch
    if specTable ~= nil and winchSpec ~= nil and winchSpec.treeRaycast ~= nil then
        if type(adjustValue) == THValueType.NUMBER and adjustValue ~= 0 then
            local objectNodeId = winchSpec.treeRaycast.lastValidTree
            local objectData = g_thBetterWinch:getAttachableObjectByComponentNode(objectNodeId)
            if objectData ~= nil then
                local _, meshData = objectData:getLastTargetData()
                if meshData ~= nil then
                    meshData.radius = THUtils.clamp(meshData.radius + adjustValue, THBetterWinch.MIN_ATTACH_RADIUS, THBetterWinch.MAX_ATTACH_RADIUS)
                    meshData.lastAxisRadius[meshData.upAxis] = meshData.radius
                end
            end
        end
    end
end
function THVSpec_Winch.onWinchAttachPointRaycastCallback(specTable, hitObjectId, wx, wy, wz, distance, dx, dy, dz, shapeIndex, hitShapeId, isLast)
    return THUtils.call(function()
        local self = specTable.vehicle
        local mission = g_thBetterWinch.coreData.mission
        local winchSpec = self.spec_winch
        if mission ~= nil and winchSpec ~= nil
            and specTable.isAttachPointRaycastMode
        then
            local raycastInfo = winchSpec.treeRaycast
            if raycastInfo ~= nil then
                raycastInfo.lastValidTree = nil
                raycastInfo.lastInValidTree = nil
                if type(hitObjectId) == THValueType.NUMBER and hitObjectId > 0
                    and type(hitShapeId) == THValueType.NUMBER and hitShapeId > 0
                    and getHasClassId(hitShapeId, ClassIds.SHAPE)
                then
                    local attachRootNode = g_thBetterWinch:getAttachPointRootNode(hitShapeId) or hitObjectId
                    local isDebugUpdateEnabled = THUtils.getIsDebugEnabled(debugFlagId, THDebugLevel.UPDATE)
                    if isDebugUpdateEnabled then
                        THUtils.displayMsg("Hit object root node: %s [%s]", getName(attachRootNode), attachRootNode)
                        THUtils.displayMsg("Hit object node: %s [%s]", getName(hitObjectId), hitObjectId)
                        THUtils.displayMsg("Hit shape node: %s [%s]", getName(hitShapeId), hitShapeId)
                        THUtils.displayMsg("Shape index: %s", shapeIndex)
                        THUtils.displayMsg("")
                    end
                    if CollisionFlag.getHasGroupFlagSet(hitShapeId, CollisionFlag.TRIGGER) then
                        if g_thBetterWinch:getAttachPointNode(hitShapeId) == nil then
                            if not isLast then
                                return true
                            end
                        end
                    end
                    local object = mission:getNodeObject(attachRootNode)
                    local rootVehicle = self:getRootVehicle() or self
                    if object ~= nil and object ~= self and object ~= rootVehicle
                        and g_thBetterWinch:validateAttachableObjectNodeId(attachRootNode)
                    then
                        local objectData = g_thBetterWinch:registerAttachableObject(object)
                        if objectData ~= nil then
                            objectData.lastTargetedComponentIndex = nil
                            objectData.lastTargetedMeshNodeIndex = nil
                            local componentData = objectData:getComponent(attachRootNode)
                            if componentData ~= nil then
                                local ownerFarmId = self:getActiveFarm()
                                local isObjectAttached = false
                                for ropeIndex = 1, #winchSpec.ropes do
                                    local ropeInfo = winchSpec.ropes[ropeIndex]
                                    for treeIndex = 1, #ropeInfo.attachedTrees do
                                        if ropeInfo.attachedTrees[treeIndex].treeId == componentData.node then
                                            isObjectAttached = true
                                        end
                                    end
                                end
                                if not isObjectAttached
                                    and mission.accessHandler:canFarmAccess(ownerFarmId, object)
                                    and (object.getIsAIActive == nil or not object:getIsAIActive())
                                then
                                    local meshData = objectData:getClosestMeshNode(wx, wy, wz, 4, false, hitShapeId)
                                    if meshData == nil then
                                        if isDebugUpdateEnabled then
                                            THUtils.displayMsg("No valid mesh node found")
                                            THUtils.displayMsg("")
                                        end
                                    else
                                        objectData.lastTargetedComponentIndex = componentData.index
                                        objectData.lastTargetedMeshNodeIndex = meshData.index
                                        if isDebugUpdateEnabled then
                                            THUtils.displayMsg("Found mesh node: %s [%s]", meshData.nodeName, meshData.node)
                                            THUtils.displayMsg("")
                                        end
                                        local cx, cy, cz, upX, upY, upZ, radius = g_thBetterWinch:getAttachableObjectOffsetPosition(componentData.node, wx, wy, wz, THBetterWinch.MAX_ATTACH_RADIUS, 0.15)
                                        if cx ~= nil then
                                            if MathUtil.vector3Length(raycastInfo.startPos[1] - wx, raycastInfo.startPos[2] - wy, raycastInfo.startPos[3] - wz) <= raycastInfo.maxDistance then
                                                raycastInfo.lastValidTree = componentData.node
                                                raycastInfo.lastInValidTree = nil
                                            else
                                                raycastInfo.lastValidTree = nil
                                                raycastInfo.lastInValidTree = componentData.node
                                            end
                                            raycastInfo.treeTargetPos[1] = wx
                                            raycastInfo.treeTargetPos[2] = wy
                                            raycastInfo.treeTargetPos[3] = wz
                                            raycastInfo.treeCenterPos[1] = cx
                                            raycastInfo.treeCenterPos[2] = cy
                                            raycastInfo.treeCenterPos[3] = cz
                                            raycastInfo.treeUp[1] = upX
                                            raycastInfo.treeUp[2] = upY
                                            raycastInfo.treeUp[3] = upZ
                                            raycastInfo.treeRadius = radius
                                            if isDebugUpdateEnabled then
                                                THUtils.displayMsg("Raycast information:")
                                                THUtils.printTable(raycastInfo, 1)
                                                THUtils.displayMsg("Current object:")
                                                THUtils.printTable(objectData)
                                                THUtils.displayMsg("Current component:")
                                                THUtils.printTable(componentData, 1)
                                                THUtils.displayMsg("Current mesh node:")
                                                THUtils.printTable(meshData, 1)
                                            end
                                            self:raiseActive()
                                        end
                                    end
                                end
                            end
                        end
                        raycastInfo.hasStarted = false
                        return false
                    end
                end
            end
        end
        if isLast then
            if winchSpec ~= nil and winchSpec.treeRaycast ~= nil
                and winchSpec.treeRaycast.hasStarted
            then
                winchSpec.treeRaycast.hasStarted = false
            end
            specTable.isAttachPointRaycastMode = false
        end
    end)
end
function THVSpec_Winch.overwrite_attachTreeToWinch(self, superFunc, objectNodeId, x, y, z, ropeIndex, ...)
    local specTable = getSpecTable(self)
    local function appendFunc(...)
        if specTable ~= nil then
            THUtils.pcall(function()
                local objectData = g_thBetterWinch:getAttachableObjectByComponentNode(objectNodeId, true)
                if objectData ~= nil then
                    objectData:updateWinchAttachData(self)
                end
            end)
        end
        return ...
    end
    return appendFunc(superFunc(self, objectNodeId, x, y, z, ropeIndex, ...))
end
function THVSpec_Winch.overwrite_detachTreeFromWinch(self, superFunc, ropeIndex, ...)
    local specTable = getSpecTable(self)
    local winchSpec = self.spec_winch
    local function appendFunc(...)
        if specTable ~= nil and winchSpec ~= nil and ropeIndex ~= nil then
            THUtils.pcall(function()
                local ropeDesc = winchSpec.ropes[ropeIndex]
                local processedObjects = THUtils.clearTable(specTable.attachedObjectsCache)
                for _, ropeAttachDesc in pairs(ropeDesc.attachedTrees) do
                    local objectData = g_thBetterWinch:getAttachableObjectByComponentNode(ropeAttachDesc.treeId, true)
                    if objectData ~= nil and not processedObjects[objectData] then
                        objectData:updateWinchAttachData(self)
                        processedObjects[objectData] = true
                    end
                end
            end)
        end
        return ...
    end
    return appendFunc(superFunc(self, ropeIndex, ...))
end
function THVSpec_Winch.overwrite_onWinchTreeRaycastCallback(self, superFunc, hitObjectId, wx, wy, wz, distance, dx, dy, dz, shapeIndex, hitShapeId, isLast, ...)
    local specTable = getSpecTable(self)
    local winchSpec = self.spec_winch
    local function appendFunc(rSuccess, ...)
        if specTable ~= nil and winchSpec ~= nil
            and (rSuccess == false or isLast)
        then
            local raycastInfo = winchSpec.treeRaycast
            if raycastInfo ~= nil
                and raycastInfo.lastValidTree == nil
                and raycastInfo.lastInValidTree == nil
            then
                specTable.isAttachPointRaycastMode = true
            end
        end
        return rSuccess, ...
    end
    return appendFunc(superFunc(self, hitObjectId, wx, wy, wz, distance, dx, dy, dz, shapeIndex, hitShapeId, isLast, ...))
end
function THVSpec_Winch.overwrite_onWinchTreeShapeMounted(self, superFunc, shapeId, otherWinch, ...)
    local specTable = getSpecTable(self)
    local winchSpec = self.spec_winch
    local noAllowRun = false
    if specTable ~= nil and winchSpec ~= nil then
        THUtils.pcall(function()
            if type(otherWinch) == THValueType.TABLE and otherWinch ~= self
                and THUtils.getIsType(shapeId, THValueType.INTEGER)
            then
                if THUtils.getNodeExists(shapeId)
                    and g_thBetterWinch:getAttachableObjectByNodeId(shapeId) ~= nil
                then
                    noAllowRun = true
                end
            end
        end)
    end
    if not noAllowRun then
        return superFunc(self, shapeId, otherWinch, ...)
    end
end
function THVSpec_Winch.gOverwrite_raycastClosestAsync(superFunc, wx, wy, wz, dx, dy, dz, distance, callbackName, callbackTarget, collisionMask, ...)
    if callbackName == "onWinchTreeRaycastCallback"
        and type(callbackTarget) == THValueType.TABLE
        and collisionMask == CollisionFlag.TREE
    then
        local self = callbackTarget
        local specTable = getSpecTable(self)
        if specTable ~= nil and specTable.isAttachPointRaycastMode then
            return raycastAllAsync(wx, wy, wz, dx, dy, dz, distance, "onWinchAttachPointRaycastCallback", specTable, THBetterWinch.COLLISION_MASK)
        end
    end
    return superFunc(wx, wy, wz, dx, dy, dz, distance, callbackName, callbackTarget, collisionMask, ...)
end
function THVSpec_Winch.gAppend_registerXMLPaths(superFunc, xmlSchema, xmlPath, ...)
    if xmlSchema ~= nil and xmlPath ~= nil then
        local xmlSchemaSavegame = Vehicle.xmlSchemaSavegame
        if xmlSchemaSavegame ~= nil then
            local xmlSubPath = "vehicles.vehicle(?)." .. g_thBetterWinch.coreData.xmlKey
            local ropePath = xmlSubPath .. ".rope(?)"
            xmlSchemaSavegame:register(XMLValueType.INT, ropePath .. "#index", "Rope internal index")
            local objectPath = ropePath .. ".attachedObject(?)"
            xmlSchemaSavegame:register(XMLValueType.STRING, objectPath .. "#uniqueId", "Object unique id")
            xmlSchemaSavegame:register(XMLValueType.INT, objectPath .. "#component", "Object component index")
            xmlSchemaSavegame:register(XMLValueType.VECTOR_TRANS, objectPath .. "#translation", "Hook current position")
            local meshNodePath = objectPath .. ".meshNode"
            xmlSchemaSavegame:register(XMLValueType.INT, meshNodePath .. "#index", "Object attach mesh index")
            xmlSchemaSavegame:register(XMLValueType.FLOAT, meshNodePath .. "#radius", "Object attach radius")
            xmlSchemaSavegame:register(XMLValueType.INT, meshNodePath .. "#upAxis", "Attach ring orientation")
            ForestryPhysicsRope.registerSavegameXMLPaths(xmlSchemaSavegame, objectPath .. ".physicsRope")
        end
    end
end
function THVSpec_Winch.gOverwrite_saveToXMLFile(self, superFunc, xmlFile, xmlKey, ...)
    local specTable = getSpecTable(self)
    local winchSpec = self.spec_winch
    local hasCachedTrees = false
    if specTable ~= nil and winchSpec ~= nil
        and xmlFile ~= nil and xmlKey ~= nil
    then
        THUtils.pcall(function()
            if winchSpec.ropes ~= nil then
                local winchRopeCache = THUtils.clearTable(specTable.winchRopeCache)
                local xmlSubKey = string.gsub(xmlKey, "%.winch.*", "." .. g_thBetterWinch.coreData.xmlKey)
                local ropeXMLIndex = 0
                for ropeIndex = 1, #winchSpec.ropes do
                    local ropeDesc = winchSpec.ropes[ropeIndex]
                    local ropeKey = string.format("%s.rope(%d)", xmlSubKey, ropeXMLIndex)
                    local isRopeEntryAdded = false
                    local cachedTrees = ropeDesc.attachedTrees
                    local objectXMLIndex = 0
                    if cachedTrees ~= nil and next(cachedTrees) ~= nil then
                        winchRopeCache[ropeIndex] = {
                            attachedTrees = cachedTrees
                        }
                        ropeDesc.attachedTrees = {}
                        for ropeAttachIndex = 1, #cachedTrees do
                            local ropeAttachDesc = cachedTrees[ropeAttachIndex]
                            local objectData = g_thBetterWinch:getAttachableObjectByComponentNode(ropeAttachDesc.treeId, true)
                            if objectData == nil then
                                table.insert(ropeDesc.attachedTrees, ropeAttachDesc)
                            elseif objectData:getIsEnabled()
                                and objectData.target ~= nil
                                and objectData.target.getUniqueId ~= nil
                                and ropeAttachDesc.activeHookData ~= nil
                                and ropeAttachDesc.activeHookData.hookId ~= nil
                                and ropeDesc.mainRope ~= nil
                            then
                                local object = objectData.target
                                local objectUniqueId = object:getUniqueId()
                                local componentData = objectData:getComponent(ropeAttachDesc.treeId)
                                if componentData ~= nil and THUtils.getNodeExists(componentData.node)
                                    and objectUniqueId ~= nil and objectUniqueId ~= ""
                                then
                                    local attachedWinchData = objectData.attachedWinches[self]
                                    if attachedWinchData ~= nil then
                                        local attachedRopeData = attachedWinchData[ropeIndex]
                                        if attachedRopeData ~= nil then
                                            local attachedComponentData = attachedRopeData[componentData.index]
                                            if attachedComponentData ~= nil then
                                                local objectKey = string.format("%s.attachedObject(%d)", ropeKey, objectXMLIndex)
                                                local meshData = attachedComponentData.meshNode
                                                if THUtils.getNodeExists(meshData.node)
                                                    and THUtils.getNodeExists(ropeAttachDesc.activeHookData.hookId)
                                                then
                                                    local hookX, hookY, hookZ = getWorldTranslation(ropeAttachDesc.activeHookData.hookId)
                                                    if not isRopeEntryAdded then
                                                        xmlFile:setValue(ropeKey .. "#index", ropeIndex)
                                                        isRopeEntryAdded = true
                                                    end
                                                    xmlFile:setValue(objectKey .. "#uniqueId", objectUniqueId)
                                                    xmlFile:setValue(objectKey .. "#component", componentData.index)
                                                    xmlFile:setValue(objectKey .. "#translation", hookX, hookY, hookZ)
                                                    local meshNodeKey = objectKey .. ".meshNode"
                                                    xmlFile:setValue(meshNodeKey .. "#index", meshData.index)
                                                    if meshData.radius ~= nil then
                                                        xmlFile:setValue(meshNodeKey .. "#radius", meshData.radius)
                                                    end
                                                    if meshData.upAxis ~= nil then
                                                        xmlFile:setValue(meshNodeKey .. "#upAxis", meshData.upAxis)
                                                    end
                                                    ropeDesc.mainRope:saveToXMLFile(xmlFile, objectKey .. ".physicsRope")
                                                    objectXMLIndex = objectXMLIndex + 1
                                                end
                                            end
                                        end
                                    end
                                end
                            end
                        end
                        hasCachedTrees = true
                        ropeXMLIndex = ropeXMLIndex + 1
                    end
                end
            end
        end)
    end
    local function appendFunc(...)
        if specTable ~= nil and winchSpec ~= nil then
            if hasCachedTrees then
                THUtils.pcall(function()
                    for ropeIndex = #specTable.winchRopeCache, 1, -1 do
                        local cacheData = specTable.winchRopeCache[ropeIndex]
                        local ropeDesc = winchSpec.ropes[ropeIndex]
                        if ropeDesc ~= nil then
                            ropeDesc.attachedTrees = cacheData.attachedTrees
                        end
                        specTable.winchRopeCache[ropeIndex] = nil
                    end
                end)
            end
        end
        return ...
    end
    return appendFunc(superFunc(self, xmlFile, xmlKey, ...))
end
THWinchAttachTreeActivatable.DATA_KEY = tostring(THWinchAttachTreeActivatable)
function THWinchAttachTreeActivatable.getCustomData(self)
    return THUtils.call(function()
        if self ~= nil then
            return THUtils.getDataTable(self, THWinchAttachTreeActivatable.DATA_KEY)
        end
    end)
end
function THWinchAttachTreeActivatable.resetActionEvents(self, inputContext)
    for eventName, eventId in pairs(self.actionEventIds) do
        g_inputBinding:removeActionEvent(eventId)
        self.actionEventIds[eventName] = nil
    end
end
function THWinchAttachTreeActivatable.updateActionEvents(self, inputContext)
    local mission = g_thBetterWinch.coreData.mission
    local vehicle = self.parent.vehicle
    local winchSpec = nil
    if vehicle ~= nil then
        winchSpec = vehicle.spec_winch
    end
    if winchSpec ~= nil and winchSpec.treeRaycast ~= nil then
        local objectData = g_thBetterWinch:getAttachableObjectByComponentNode(winchSpec.treeRaycast.lastValidTree)
        local meshData = nil
        if objectData ~= nil then
            meshData = objectData:getMeshNodeByIndex(objectData.lastTargetedMeshNodeIndex)
        end
        for _, eventId in pairs(self.actionEventIds) do
            local isEventActive = true
            if eventId == self.actionEventIds.adjustAttachRadius
                or eventId == self.actionEventIds.rotateAttachLoop
            then
                if meshData == nil then
                    isEventActive = false
                end
            end
            g_inputBinding:setActionEventTextVisibility(eventId, isEventActive)
            g_inputBinding:setActionEventActive(eventId, isEventActive)
            if isEventActive and mission ~= nil then
                if meshData ~= nil then
                    if eventId == self.actionEventIds.adjustAttachRadius then
                        local helpText = g_i18n:getText("thBetterWinch_currentAttachRadius")
                        helpText = string.format(helpText, meshData.radius)
                        mission:addExtraPrintText(helpText)
                    elseif eventId == self.actionEventIds.rotateAttachLoop then
                        local attachAxisText = nil
                        if meshData.upAxis == THAxis.X then
                            attachAxisText = "X"
                        elseif meshData.upAxis == THAxis.Y then
                            attachAxisText = "Y"
                        elseif meshData.upAxis == THAxis.Z then
                            attachAxisText = "Z"
                        end
                        if attachAxisText ~= nil then
                            local helpText = g_i18n:getText("thBetterWinch_currentAttachAxis")
                            helpText = string.format(helpText, attachAxisText)
                            mission:addExtraPrintText(helpText)
                        end
                    end
                end
            end
        end
    else
        for _, eventId in pairs(self.actionEventIds) do
            g_inputBinding:setActionEventTextVisibility(eventId, false)
            g_inputBinding:setActionEventActive(eventId, false)
        end
    end
end
function THWinchAttachTreeActivatable.onActionAdjustAttachRadius(self, actionName, inputValue, callbackState, isAnalog)
    THUtils.pcall(function()
        local vehicle = self.parent.vehicle
        local specTable = getSpecTable(vehicle)
        if specTable ~= nil then
            if inputValue ~= self.lastAttachRadiusDirection then
                self.lastAttachRadiusTimer = 0
                self.lastAttachRadiusDirection = inputValue
                self.attachRadiusAccelTimer = 0
                self.attachRadiusAccelFactor = 1
            else
                self.lastAttachRadiusTimer = 500
            end
            if inputValue ~= 0 then
                local adjustValue = 0.001 * inputValue * self.attachRadiusAccelFactor
                THVSpec_Winch.adjustWinchAttachRadius(vehicle, adjustValue)
            end
        end
    end)
end
function THWinchAttachTreeActivatable.onActionRotateAttachLoop(self, actionName, inputValue, callbackState, isAnalog)
    THUtils.pcall(function()
        local vehicle = self.parent.vehicle
        local specTable = getSpecTable(vehicle)
        local winchSpec = vehicle.spec_winch
        if specTable ~= nil and winchSpec ~= nil
            and winchSpec.treeRaycast ~= nil
            and winchSpec.treeRaycast.lastValidTree ~= nil
        then
            local objectData = g_thBetterWinch:getAttachableObjectByComponentNode(winchSpec.treeRaycast.lastValidTree)
            if objectData ~= nil then
                local meshData = objectData:getMeshNodeByIndex(objectData.lastTargetedMeshNodeIndex)
                if meshData ~= nil then
                    local upAxis = meshData.upAxis + 1
                    if upAxis > THAxis.Z then
                        upAxis = THAxis.X
                    end
                    meshData.radius = meshData.lastAxisRadius[upAxis] + 0.01
                    meshData.upAxis = upAxis
                end
            end
        end
    end)
end
function THWinchAttachTreeActivatable.gAppend_registerCustomInput(superFunc, parent, inputContext, ...)
    local self = THWinchAttachTreeActivatable.getCustomData(parent)
    if self ~= nil then
        self:resetActionEvents(inputContext)
        local _, eventId = g_inputBinding:registerActionEvent(InputAction.TH_BETTER_WINCH_ADJUST_RADIUS, self, THWinchAttachTreeActivatable.onActionAdjustAttachRadius, false, false, true, true)
        if eventId ~= nil then
            g_inputBinding:setActionEventTextPriority(eventId, GS_PRIO_VERY_HIGH)
            g_inputBinding:setActionEventText(eventId, THVSpec_Winch.ACTION_TEXT.ADJUST_ATTACH_RADIUS)
        end
        self.actionEventIds.adjustAttachRadius = eventId
        _, eventId = g_inputBinding:registerActionEvent(InputAction.TH_BETTER_WINCH_ROTATE_LOOP, self, THWinchAttachTreeActivatable.onActionRotateAttachLoop, false, true, false, true)
        if eventId ~= nil then
            g_inputBinding:setActionEventTextPriority(eventId, GS_PRIO_VERY_HIGH)
            g_inputBinding:setActionEventText(eventId, THVSpec_Winch.ACTION_TEXT.ROTATE_ATTACH_LOOP)
        end
        self.actionEventIds.rotateAttachLoop = eventId
        self:updateActionEvents(inputContext)
    end
end
function THWinchAttachTreeActivatable.gAppend_removeCustomInput(superFunc, parent, inputContext, ...)
    local self = THWinchAttachTreeActivatable.getCustomData(parent)
    if self ~= nil then
        self:resetActionEvents(inputContext)
    end
end
function THWinchAttachTreeActivatable.gAppend_update(superFunc, parent, dt, ...)
    local self = THWinchAttachTreeActivatable.getCustomData(parent)
    if self ~= nil then
        self:updateActionEvents()
        if self.lastAttachRadiusTimer > 0 then
            if self.attachRadiusAccelTimer >= THBetterWinch.ATTACH_RADIUS_ACCEL_DELAY then
                self.attachRadiusAccelFactor = math.max(1, self.attachRadiusAccelFactor + (0.01 * dt))
            else
                self.attachRadiusAccelTimer = self.attachRadiusAccelTimer + dt
            end
            self.lastAttachRadiusTimer = self.lastAttachRadiusTimer - dt
        else
            self.attachRadiusAccelTimer = 0
            self.attachRadiusAccelFactor = 1
        end
    end
end
function THWinchAttachTreeActivatable.gOverwrite_new(superFunc, vehicle, rope, ...)
    local function appendFunc(rActivatable, ...)
        if rActivatable ~= nil then
            THUtils.pcall(function()
                local self = THUtils.createDataTable(rActivatable, THWinchAttachTreeActivatable.DATA_KEY, THWinchAttachTreeActivatable_mt)
                if self ~= nil then
                    self.actionEventIds = {}
                    self.lastAttachRadiusTimer = 0
                    self.lastAttachRadiusDirection = 0
                    self.attachRadiusAccelTimer = 0
                    self.attachRadiusAccelFactor = 1
                end
            end)
        end
        return rActivatable, ...
    end
    return appendFunc(superFunc(vehicle, rope, ...))
end
THUtils.pcall(initScript)