EAS_AnimalSystem = {}

EAS_AnimalSystem.cowsPath = g_currentModDirectory .. "xmls/cows.xml"
EAS_AnimalSystem.pigsPath = g_currentModDirectory .. "xmls/pigs.xml"
EAS_AnimalSystem.sheepsPath = g_currentModDirectory .. "xmls/sheeps.xml"
EAS_AnimalSystem.horsesPath = g_currentModDirectory .. "xmls/horses.xml"
EAS_AnimalSystem.chickensPath = g_currentModDirectory .. "xmls/chickens.xml"
EAS_AnimalSystem.duckwilds = g_currentModDirectory .. "xmls/duckwilds.xml"
EAS_AnimalSystem.rabbits = g_currentModDirectory .. "xmls/rabbits.xml"
EAS_AnimalSystem.goose = g_currentModDirectory .. "xmls/goose.xml"
EAS_AnimalSystem.animalsPath = {
    EAS_AnimalSystem.cowsPath,
    EAS_AnimalSystem.pigsPath,
    EAS_AnimalSystem.sheepsPath,
    EAS_AnimalSystem.horsesPath,
    EAS_AnimalSystem.chickensPath,
    EAS_AnimalSystem.duckwilds,
    EAS_AnimalSystem.rabbits,
    EAS_AnimalSystem.goose
}

EAS_AnimalSystem.defaultInseminationCost = 15
EAS_AnimalSystem.defaultMaxAge = 150
EAS_AnimalSystem.defaultMaxAgeDiePercentage = 5
EAS_AnimalSystem.defaultHealthEmptyDiePercentage = 3
EAS_AnimalSystem.defaultInseminationProbabilitiesTooOldAge = 180
EAS_AnimalSystem.defaultInseminationProbabilitiesSteps = 8

local EAS_AnimalSystem_mt = Class(EAS_AnimalSystem)

function EAS_AnimalSystem.new(customMt)
	local self = setmetatable({}, customMt or EAS_AnimalSystem_mt)
	self.isServer = false
	self.mission = false

    self.originalAnimals = {}
    self.animals = {}

    for _, path in pairs(EAS_AnimalSystem.animalsPath) do
        EAS_AnimalSystem.loadAnimalsXml(self, path)
    end

	return self
end

function EAS_AnimalSystem.loadAnimalsXml(self, path)
	local xmlFile = XMLFile.wrap(loadXMLFile("animals", path))

    xmlFile:iterate("animals.animal", function (_, animalTypeKey)
        local animal = EAS_AnimalSystem.loadAnimalTypeValues(animalTypeKey, xmlFile)
        local subTypes = {}

        xmlFile:iterate(animalTypeKey..".subType", function (_, subTypeKey)
            local subType = EAS_AnimalSystem.loadSubTypeValues(subTypeKey, xmlFile)
            subTypes[xmlFile:getString(subTypeKey .. "#subType")] = subType
        end)

        animal.subTypes = subTypes

        self.animals[xmlFile:getString(animalTypeKey .. "#type")] = animal

    end)

    xmlFile:delete()
end

function EAS_AnimalSystem.loadAnimalTypeValues(animalTypeKey, xmlFile)
    local animal = {
        inseminationCost = xmlFile:getInt(animalTypeKey .. "#inseminationCost"),
        maxAge = xmlFile:getInt(animalTypeKey .. "#maxAge"),
        maxAgeDiePercentage = xmlFile:getInt(animalTypeKey .. "#maxAgeDiePercentage"),
        healthEmptyDiePercentage = xmlFile:getInt(animalTypeKey .. "#healthEmptyDiePercentage")
    }

    return animal
end

function EAS_AnimalSystem.loadSubTypeValues(subTypeKey, xmlFile)
    local subType = {
        reproductionProbabilities = {},
        inseminationProbabilities = {},
        lactation = {},
        inseminationTooOldAge = xmlFile:getInt(subTypeKey .. ".inseminationProbabilities#tooOldAge"),
        inseminationTooOldAgeSteps = xmlFile:getInt(subTypeKey .. ".inseminationProbabilities#steps"),
        malePart = xmlFile:getString(subTypeKey .. "#malePart")
    }

    xmlFile:iterate(subTypeKey..".reproduction.key", function (_, reproductionKey)
        local reproductionValue = {
            numOfAnimals = xmlFile:getInt(reproductionKey .. "#numOfAnimals"),
            probability = xmlFile:getInt(reproductionKey .. "#probability")
        }

        table.insert(subType.reproductionProbabilities, reproductionValue)
    end)

    xmlFile:iterate(subTypeKey..".inseminationProbabilities.key", function (_, reproductionKey)
        local inseminationProbabilityValue = {
            monthSinceLastBirth = xmlFile:getInt(reproductionKey .. "#monthSinceLastBirth"),
            probability = xmlFile:getInt(reproductionKey .. "#probability")
        }

        table.insert(subType.inseminationProbabilities, inseminationProbabilityValue)
    end)

    xmlFile:iterate(subTypeKey..".lactation.key", function (_, lactationKey)
        local lactationValue = {
            month = xmlFile:getInt(lactationKey .. "#month"),
            food = xmlFile:getFloat(lactationKey .. "#food"),
            milk = xmlFile:getFloat(lactationKey .. "#milk")
        }

        table.insert(subType.lactation, lactationValue)
    end)

    if #subType.reproductionProbabilities == 0 then
        subType.reproductionProbabilities = nil
    end

    if #subType.inseminationProbabilities == 0 then
        subType.inseminationProbabilities = nil
    end

    if #subType.lactation == 0 then
        subType.lactation = nil
    end

    return subType
end

function EAS_AnimalSystem:addSubTypeProperties(animalTypName, subType, originalSubTypeValues)
    local easAnimalType = self.animals[animalTypName]

    if easAnimalType == nil then
        return
    end

    local easSubType = easAnimalType.subTypes[subType.name]
    local defaultSubType = easAnimalType.subTypes["default"]

    if easSubType == nil then
        if defaultSubType == nil then
            return
        else
            easSubType = defaultSubType
        end
    end

    subType.malePart = originalSubTypeValues.malePart or easSubType.malePart or defaultSubType.malePart
    subType.lactation = originalSubTypeValues.lactation or easSubType.lactation or defaultSubType.lactation
    subType.inseminationTooOldAge = originalSubTypeValues.inseminationTooOldAge or easSubType.inseminationTooOldAge or defaultSubType.inseminationTooOldAge
    subType.inseminationTooOldAgeSteps = originalSubTypeValues.inseminationTooOldAgeSteps or easSubType.inseminationTooOldAgeSteps or defaultSubType.inseminationTooOldAgeSteps

    self:addReproductionProbabilities(subType, easSubType, defaultSubType, originalSubTypeValues)
    self:addInseminationProbabilities(subType, easSubType, defaultSubType, originalSubTypeValues)
end

function EAS_AnimalSystem:addReproductionProbabilities(subType, easSubType, defaultSubType, originalSubTypeValues)
    subType.reproductionProbabilities = easSubType.reproductionProbabilities
    local repoductionProbabilitiesMap = {}
    local reproductionProbabilities = originalSubTypeValues.reproductionProbabilities or easSubType.reproductionProbabilities or defaultSubType.reproductionProbabilities
    for _, probabilityValue in pairs(reproductionProbabilities) do
        for i = 1, probabilityValue.probability do
            table.insert(repoductionProbabilitiesMap, probabilityValue.numOfAnimals)
        end
    end
    subType.repoductionProbabilitiesMap = repoductionProbabilitiesMap
end

function EAS_AnimalSystem:addInseminationProbabilities(subType, easSubType, defaultSubType, originalSubTypeValues)
    subType.inseminationProbabilities = easSubType.inseminationProbabilities
    local inseminationProbabilitiesMap = {}
    local inseminationProbabilities = originalSubTypeValues.inseminationProbabilities or easSubType.inseminationProbabilities or defaultSubType.inseminationProbabilities
    for _, inseminationProbabilityValue in pairs(inseminationProbabilities) do
        inseminationProbabilitiesMap[inseminationProbabilityValue.monthSinceLastBirth] = inseminationProbabilityValue.probability
    end
    subType.inseminationProbabilitiesMap = inseminationProbabilitiesMap
end

function EAS_AnimalSystem:addAnimalTypeProperties(animalType, originalAnimalTypeValues)
    local easAnimalType = self.animals[animalType.name]

    if easAnimalType == nil then
        animalType.inseminationCost = originalAnimalTypeValues.inseminationCost or EAS_AnimalSystem.defaultInseminationCost
        animalType.maxAge = originalAnimalTypeValues.maxAge or EAS_AnimalSystem.defaultMaxAge
        animalType.maxAgeDiePercentage = originalAnimalTypeValues.maxAgeDiePercentage or EAS_AnimalSystem.defaultMaxAgeDiePercentage
        animalType.healthEmptyDiePercentage = originalAnimalTypeValues.healthEmptyDiePercentage or EAS_AnimalSystem.defaultHealthEmptyDiePercentage
    else
        animalType.inseminationCost = originalAnimalTypeValues.inseminationCost or easAnimalType.inseminationCost
        animalType.maxAge = originalAnimalTypeValues.maxAge or easAnimalType.maxAge
        animalType.maxAgeDiePercentage = originalAnimalTypeValues.maxAgeDiePercentage or easAnimalType.maxAgeDiePercentage
        animalType.healthEmptyDiePercentage = originalAnimalTypeValues.healthEmptyDiePercentage or easAnimalType.healthEmptyDiePercentage
    end
end

------------------------------------------------------

function EAS_AnimalSystem.loadSubType(self, superFunc, animalType, subType, xmlFile, subTypeKey, baseDirectory)
    local returnValue = superFunc(self, animalType, subType, xmlFile, subTypeKey, baseDirectory)
    local originalSubTypeValues = EAS_AnimalSystem.loadSubTypeValues(subTypeKey, xmlFile)
    g_eas_animalSystem:addSubTypeProperties(animalType.name, subType, originalSubTypeValues)
    subType.reproductionMinHealth = 0
    return returnValue
end

AnimalSystem.loadSubType = Utils.overwrittenFunction(AnimalSystem.loadSubType, EAS_AnimalSystem.loadSubType)

function EAS_AnimalSystem.loadSubTypes(self, superFunc, animalType, xmlFile, animalTypeKey, baseDirectory)
    local returnValue = superFunc(self, animalType, xmlFile, animalTypeKey, baseDirectory)
    local originalAnimalTypeValues = EAS_AnimalSystem.loadAnimalTypeValues(animalTypeKey, xmlFile)
    g_eas_animalSystem:addAnimalTypeProperties(animalType, originalAnimalTypeValues)
    return returnValue
end

AnimalSystem.loadSubTypes = Utils.overwrittenFunction(AnimalSystem.loadSubTypes, EAS_AnimalSystem.loadSubTypes)

g_eas_animalSystem = EAS_AnimalSystem.new()