# HistoricalPricing Development Guide

## Current Status

### ✅ Completed Features
- **FillType Prices**: Crop/product selling prices working correctly
- **Selling Station Hook**: Successfully hooks into `SellingStation.getEffectiveFillTypePrice`
- **Field Prices Hook**: Hooks into `Farmland.updatePrice` to modify land purchase costs
- **Settings GUI**: In-game menu for year selection
- **Data Separation**: Split into `fillTypePrices.xml` and `fieldPrices.xml`
- **Settings Persistence**: Saves selected year to savegame directory

### 🔄 In Progress
- Field price testing (hook installed, needs in-game verification)

### ❌ Not Yet Implemented
- Buying station prices (seeds, fertilizer, etc.)
- Multiplayer support (requires zipping mod)
- Clean up debug logging

## Architecture

### Core Components

#### TTYPricing.lua
Main mod logic with the following key functions:

- `loadMap()` - Initializes mod, loads data files
- `update(dt)` - Handles delayed initialization and month/year tracking
- `loadFillTypePrices()` - Loads crop price data from XML
- `loadFieldPrices()` - Loads land price data from XML
- `updatePrices()` - Applies crop price multipliers
- `setupSellingStationHook()` - Hooks into selling price calculation
- `setupFarmlandPriceHook()` - Hooks into field price calculation
- `loadSettings()` / `saveSettings()` - Persistence layer

#### TTYPricingSettings.lua
In-game settings GUI:

- Dynamically loads years from pricing data
- Integrates with FS25's settings menu
- Calls `TTYPricing:updatePrices()` when year changes

### Data Structure

#### fillTypePrices.xml
```xml
<ttyPricing>
    <year value="1992">
        <crop name="WHEAT" price="0.63" />
        <crop name="BARLEY" price="0.53" />
        <!-- price = multiplier applied to base game price -->
    </year>
</ttyPricing>
```

#### fieldPrices.xml
```xml
<ttyPricing>
    <historicalLandPrices>
        <year value="1992" price="0.44" />
        <!-- price = multiplier applied to base game land price -->
    </historicalLandPrices>
</ttyPricing>
```

## Key Technical Decisions

### Why Hook Instead of Direct Modification?

**Selling Prices**: `fillType.economy.pricePerLiter` doesn't exist in FS25. The economy system uses `SellingStation.getEffectiveFillTypePrice()` to calculate prices dynamically.

**Field Prices**: `Farmland.getPrice()` doesn't exist. The correct method is `Farmland.updatePrice()` which sets `self.price`.

### Delayed Initialization

The mod waits for `g_fillTypeManager` and savegame directory to be available before loading settings and applying prices. This prevents errors during game startup.

### Original Price Storage

We store `ttyOriginalPrice` on each object to avoid compounding multipliers when the year changes multiple times.

## Known Issues

### Non-Critical
1. **Settings save warning on initialization**: `currentMonth` is nil during first save attempt
   - **Impact**: Warning message in log, but doesn't affect functionality
   - **Fix**: Add initialization check before first save

### To Investigate
1. **Field prices**: Hook is installed but needs in-game testing
2. **Buying prices**: Not yet implemented

## Development Workflow

### Making Changes

1. **Edit Lua files** in `scripts/`
2. **Edit data files** (`fillTypePrices.xml`, `fieldPrices.xml`)
3. **Test in-game**:
   - Restart FS25 completely (Lua changes require restart)
   - Check log file for errors: `~/Library/Application Support/FarmingSimulator2025/log.txt`
4. **Commit changes** with descriptive messages

### Testing Checklist

- [ ] Crop prices update when changing year
- [ ] Field prices update when changing year
- [ ] Settings persist across game restarts
- [ ] Year auto-advances in December→January (if enabled)
- [ ] No errors in log file
- [ ] Multiplayer compatibility (if zipped)

### Adding New Years

1. Add data to `fillTypePrices.xml`:
   ```xml
   <year value="2025">
       <crop name="WHEAT" price="1.05" />
       <!-- ... -->
   </year>
   ```

2. Add data to `fieldPrices.xml`:
   ```xml
   <year value="2025" price="1.10" />
   ```

3. Settings GUI will automatically detect new years

## API Reference

### FS25 APIs Used

- `g_fillTypeManager` - Access to fill type data
- `g_currentMission.missionInfo.savegameDirectory` - Save location
- `g_currentMission.environment.currentMonth` - Current game month
- `SellingStation.getEffectiveFillTypePrice()` - Sell price calculation
- `Farmland.updatePrice()` - Field price calculation
- `Utils.overwrittenFunction()` - Function hooking
- XML functions: `loadXMLFile`, `createXMLFile`, `getXMLInt`, `getXMLFloat`, etc.

### Mod Events

- `loadMap(name)` - Called when map loads
- `update(dt)` - Called every frame
- `deleteMap()` - Called when map unloads (not currently used)

## Future Enhancements

### Planned Features
1. **Buying Station Prices**: Hook into `BuyingStation.getEffectiveFillTypePrice`
2. **Equipment Prices**: Historical vehicle/implement pricing
3. **Wage Costs**: Adjust worker wages by year
4. **Inflation Calculator**: Show equivalent modern values

### Code Cleanup
1. Remove debug logging from production
2. Add error handling for missing data
3. Optimize XML loading
4. Add unit tests (if possible in FS25 modding)

## Debugging Tips

### Enable Debug Logging
Already enabled in current version. Look for lines starting with:
```
--> Historical Pricing:
```

### Common Issues

**Prices not updating**:
- Check log for "Price multipliers updated for [year]"
- Verify year exists in XML files
- Ensure game restarted after code changes

**Hook not installing**:
- Check for "hook installed successfully" messages
- Verify FS25 version compatibility
- Check for conflicts with other mods

**Settings not saving**:
- Verify savegame directory exists
- Check file permissions
- Look for XML save errors in log

## Resources

- [FS25 Lua Documentation](https://gdn.giants-software.com/documentation.php)
- [Giants Developer Network](https://gdn.giants-software.com/)
- [FS25 Modding Discord](https://discord.gg/farming-simulator)

## Contributing

When contributing, please:
1. Follow existing code style
2. Add comments for complex logic
3. Update this document with any architectural changes
4. Test thoroughly before committing
5. Write descriptive commit messages

## Contact

For questions or issues, contact Roel Zeilstra or open an issue on the repository.
