HistoricalPricing = {}
HistoricalPricing.modDirectory = g_currentModDirectory

-- Pricing data is now loaded from XML files

function HistoricalPricing:loadMap(name)
    print("--> Historical Pricing: Mod Loaded")
    
    -- Initialize tracking variables
    self.currentMonth = nil
    self.customYear = nil
    self.isInitialized = false
    self.initializationAttempts = 0
    self.isClient = not g_currentMission:getIsServer()
    
    -- Load pricing data from XML files (this is safe to do early)
    self:loadFillTypePrices()
    self:loadProductPrices()
    self:loadFieldPrices()
    
    -- Load settings immediately (all clients need this for UI initialization)
    -- Server's year will be synced to clients via HistoricalPricingYearEvent
    HistoricalPricing.readSettings()
    
    -- Defer price updates until game is fully initialized
    print("--> Historical Pricing: Waiting for game initialization...")
    
    -- Register console commands for testing
    addConsoleCommand("hpSetYear", "Sets the historical pricing year (Usage: hpSetYear 2025)", "consoleCommandSetYear", self)
    addConsoleCommand("hpNextYear", "Advances to the next year", "consoleCommandNextYear", self)
    addConsoleCommand("hpSyncLogin", "Simulates the join synchronization for testing", "consoleCommandSyncLogin", self)
end

-- OLD SETTINGS METHODS REMOVED - NOW HANDLED BY HistoricalPricingSettings.lua

function HistoricalPricing:update(dt)
    -- Handle delayed initialization
    if not self.isInitialized then
        self.initializationAttempts = self.initializationAttempts + 1
        
        -- Check if game is ready (fillTypeManager, Station classes, and Farmland available)
        -- We wait for these classes to be defined to ensure hooks can be installed
        -- We specifically check if farmlands are loaded (next(...) not nil) to avoid empty initialization
        if g_fillTypeManager and SellingStation and BuyingStation and Farmland and g_farmlandManager and g_farmlandManager.farmlands and next(g_farmlandManager.farmlands) then
            print("--> Historical Pricing: Game fully initialized (including Farmlands), loading settings and applying prices...")
            
            -- Now it's safe to update prices
            
            -- Install hooks BEFORE updating prices so that updateFieldPrices() triggers the hooked function
            self:setupSellingStationHook()
            self:setupBuyingStationHook()
            self:setupFarmlandPriceHook()
            
            -- Apply prices (this triggers field price updates using the installed hook)
            self:updatePrices()
            
            -- Save settings (ensure file exists/updates)
            if not self.isClient then
                 HistoricalPricing.writeSettings()
            end
            
            self.isInitialized = true
            print("--> Historical Pricing: Initialization complete after " .. self.initializationAttempts .. " attempts")
        elseif self.initializationAttempts > 600 then
            -- Fallback after ~10 seconds (assuming 60 FPS) to prevent infinite wait
            -- If stations are still missing, we proceed (hooks will failed safely inside setup functions)
            print("--> Historical Pricing: WARNING - Initialization timeout (waiting for Classes), proceeding anyway...")
            self:updatePrices()
            self:setupSellingStationHook()
            self:setupBuyingStationHook()
            self:setupFarmlandPriceHook()
            self.isInitialized = true
        end
        
        return -- Don't process month changes until initialized
    end

    
    -- Check if month has changed
    local newMonth = g_currentMission.environment.currentMonth
    
    if self.currentMonth and newMonth and newMonth ~= self.currentMonth then
        print("--> Historical Pricing: Month changed from " .. tostring(self.currentMonth) .. " to " .. tostring(newMonth))
        
        -- Check if year wrapped around (e.g., 12 -> 1, or 11 -> 1 if December was skipped)
        if newMonth < self.currentMonth then
            print("--> Historical Pricing: Year rollover detected (Month " .. self.currentMonth .. " -> " .. newMonth .. ")")
            
            -- Only increment year on server
            if g_currentMission:getIsServer() then
                self.customYear = self.customYear + 1
                print("--> Historical Pricing: Year incremented to " .. tostring(self.customYear))
                
                -- Save the new year to settings
                HistoricalPricing.writeSettings()
                
                -- Update prices for the new year
                self:updatePrices()
                
                -- Broadcast to all clients
                if HistoricalPricingYearEvent then
                    HistoricalPricingYearEvent.sendEvent(self.customYear)
                end
            end
        end
        
        -- Update current month and save
        self.currentMonth = newMonth
        if g_currentMission:getIsServer() then
             HistoricalPricing.writeSettings()
        end
    end
end

function HistoricalPricing:loadFillTypePrices()
    self.pricingData = {}
    local xmlFile = loadXMLFile("FillTypePricesXML", self.modDirectory .. "fillTypePrices.xml")
    
    if xmlFile == 0 then
        print("--> Historical Pricing: Could not load fillTypePrices.xml")
        return
    end

    local i = 0
    while true do
        local key = string.format("HistoricalCropPrices.year(%d)", i)
        if not hasXMLProperty(xmlFile, key) then break end

        local year = getXMLInt(xmlFile, key .. "#value")
        if year then
            self.pricingData[year] = {}
            local j = 0
            while true do
                local cropKey = string.format("%s.crop(%d)", key, j)
                if not hasXMLProperty(xmlFile, cropKey) then break end

                local name = getXMLString(xmlFile, cropKey .. "#name")
                local price = getXMLFloat(xmlFile, cropKey .. "#price")
                
                if name and price then
                    self.pricingData[year][name] = price
                end
                j = j + 1
            end
        end
        i = i + 1
    end
    
    delete(xmlFile)
    print("--> Historical Pricing: FillType prices loaded from fillTypePrices.xml")
end

function HistoricalPricing:loadProductPrices()
    self.productPricingData = {}
    local xmlFile = loadXMLFile("ProductPricesXML", self.modDirectory .. "productPrices.xml")
    
    if xmlFile == 0 then
        print("--> Historical Pricing: Could not load productPrices.xml")
        return
    end

    local i = 0
    while true do
        local key = string.format("historicalProductPrices.year(%d)", i)
        if not hasXMLProperty(xmlFile, key) then break end

        local year = getXMLInt(xmlFile, key .. "#value")
        if year then
            self.productPricingData[year] = {}
            local j = 0
            while true do
                local inputKey = string.format("%s.input(%d)", key, j)
                if not hasXMLProperty(xmlFile, inputKey) then break end

                local name = getXMLString(xmlFile, inputKey .. "#name")
                local price = getXMLFloat(xmlFile, inputKey .. "#price")
                
                if name and price then
                    self.productPricingData[year][name] = price
                end
                j = j + 1
            end
        end
        i = i + 1
    end
    
    delete(xmlFile)
    print("--> Historical Pricing: Product prices loaded from productPrices.xml")
end

function HistoricalPricing:loadFieldPrices()
    self.fieldPriceMultipliers = {}
    local xmlFile = loadXMLFile("FieldPricesXML", self.modDirectory .. "fieldPrices.xml")
    
    if xmlFile == 0 then
        print("--> Historical Pricing: Could not load fieldPrices.xml")
        return
    end

    local i = 0
    while true do
        local key = string.format("historicalLandPrices.year(%d)", i)
        if not hasXMLProperty(xmlFile, key) then break end

        local year = getXMLInt(xmlFile, key .. "#value")
        local price = getXMLFloat(xmlFile, key .. "#price")
        
        if year and price then
            self.fieldPriceMultipliers[year] = price
        end
        
        i = i + 1
    end
    
    delete(xmlFile)
    print("--> Historical Pricing: Field prices loaded from fieldPrices.xml (" .. i .. " years)")
end

function HistoricalPricing:updatePrices()
    if not self.pricingData or not self.customYear then return end

    print("--> Historical Pricing: Using custom year: " .. tostring(self.customYear))

    -- Update crop/selling prices
    local yearData = self.pricingData[self.customYear]
    
    if yearData then
        print("--> Historical Pricing: Found crop pricing data for year " .. self.customYear)
        
        -- Store the multipliers for use in the selling station hook
        self.priceMultipliers = {}
        
        local invalidFillTypes = {}
        
        for fillTypeName, multiplier in pairs(yearData) do
            local fillType = g_fillTypeManager:getFillTypeByName(fillTypeName)
            if fillType then
                -- Store multiplier by fillType index for the selling station hook
                self.priceMultipliers[fillType.index] = multiplier
                
                print(string.format("--> Historical Pricing: Set %s sell multiplier to %.2f", fillTypeName, multiplier))
            else
                table.insert(invalidFillTypes, fillTypeName)
            end
        end
        
        -- Report invalid fill types once at the end
        if #invalidFillTypes > 0 then
            print("--> Historical Pricing: WARNING - Invalid crop fillTypes: " .. table.concat(invalidFillTypes, ", "))
        end
        
        print("--> Historical Pricing: Crop price multipliers updated for " .. self.customYear)
    else
        print("--> Historical Pricing: No crop pricing data for year " .. self.customYear .. ". Using default prices.")
        self.priceMultipliers = {}
    end
    
    -- Update product/buying prices
    local productYearData = self.productPricingData and self.productPricingData[self.customYear]
    
    if productYearData then
        print("--> Historical Pricing: Found product pricing data for year " .. self.customYear)
        
        -- Store the multipliers for use in the buying station hook
        self.productPriceMultipliers = {}
        
        local invalidProductTypes = {}
        
        for fillTypeName, multiplier in pairs(productYearData) do
            local fillType = g_fillTypeManager:getFillTypeByName(fillTypeName)
            if fillType then
                -- Store multiplier by fillType index for the buying station hook
                self.productPriceMultipliers[fillType.index] = multiplier
                
                print(string.format("--> Historical Pricing: Set %s buy multiplier to %.2f", fillTypeName, multiplier))
            else
                table.insert(invalidProductTypes, fillTypeName)
            end
        end
        
        -- Report invalid product types once at the end
        if #invalidProductTypes > 0 then
            print("--> Historical Pricing: WARNING - Invalid product fillTypes: " .. table.concat(invalidProductTypes, ", "))
        end
        
        print("--> Historical Pricing: Product price multipliers updated for " .. self.customYear)
    else
        print("--> Historical Pricing: No product pricing data for year " .. self.customYear .. ". Using default prices.")
        self.productPriceMultipliers = {}
    end
    
    -- Update field prices
    self:updateFieldPrices()
end

-- Force all farmlands to recalculate their prices
function HistoricalPricing:updateFieldPrices()
    if not g_farmlandManager or not g_farmlandManager.farmlands then
        print("--> Historical Pricing: Cannot update field prices - farmland manager not available")
        return
    end
    
    local count = 0
    for _, farmland in pairs(g_farmlandManager.farmlands) do
        if farmland.updatePrice then
            farmland:updatePrice()
            count = count + 1
        end
    end
    
    print(string.format("--> Historical Pricing: Updated %d field prices", count))
end

-- Hook into SellingStation to modify sell prices
function HistoricalPricing:setupSellingStationHook()
    if SellingStation and SellingStation.getEffectiveFillTypePrice then
        print("--> Historical Pricing: Installing selling station price hook")
        
        SellingStation.getEffectiveFillTypePrice = Utils.overwrittenFunction(
            SellingStation.getEffectiveFillTypePrice, 
            function(self, superFunc, fillTypeIndex, ...)
                local basePrice = superFunc(self, fillTypeIndex, ...)
                
                -- Apply our year-based multiplier if we have one for this fillType
                if HistoricalPricing.priceMultipliers and HistoricalPricing.priceMultipliers[fillTypeIndex] then
                    local multiplier = HistoricalPricing.priceMultipliers[fillTypeIndex]
                    return basePrice * multiplier
                end
                
                return basePrice
            end
        )
        
        print("--> Historical Pricing: Selling station price hook installed successfully")
    else
        print("--> Historical Pricing: WARNING - Could not install selling station hook")
    end
end

-- Hook into BuyingStation to modify buy prices for products (Store)
function HistoricalPricing:setupBuyingStationHook()
    if BuyingStation and BuyingStation.getEffectiveFillTypePrice then
        print("--> Historical Pricing: Installing buying station price hook")
        
        BuyingStation.getEffectiveFillTypePrice = Utils.overwrittenFunction(
            BuyingStation.getEffectiveFillTypePrice, 
            function(self, superFunc, fillTypeIndex, ...)
                local basePrice = superFunc(self, fillTypeIndex, ...)
                
                -- Apply our year-based multiplier if we have one for this fillType
                if HistoricalPricing.productPriceMultipliers and HistoricalPricing.productPriceMultipliers[fillTypeIndex] then
                    local multiplier = HistoricalPricing.productPriceMultipliers[fillTypeIndex]
                    return basePrice * multiplier
                end
                
                return basePrice
            end
        )
        
        print("--> Historical Pricing: Buying station price hook installed successfully")
    else
        print("--> Historical Pricing: WARNING - Could not install buying station hook")
    end
end

-- Hook into Farmland to modify field purchase prices
function HistoricalPricing:setupFarmlandPriceHook()
    if Farmland and Farmland.updatePrice then
        print("--> Historical Pricing: Installing farmland price hook")
        
        -- Store original updatePrice
        local originalUpdatePrice = Farmland.updatePrice
        
        -- Override updatePrice to apply our multiplier
        Farmland.updatePrice = function(self, ...)
            -- Call original updatePrice first
            originalUpdatePrice(self, ...)
            
            -- Now modify the price with our multiplier
            if HistoricalPricing.customYear and HistoricalPricing.fieldPriceMultipliers and self.price then
                local multiplier = HistoricalPricing.fieldPriceMultipliers[HistoricalPricing.customYear]
                
                if multiplier then
                    -- Store original price if not already stored
                    if not self.originalPrice then
                        self.originalPrice = self.price
                    end
                    
                    -- Apply multiplier to original price
                    self.price = self.originalPrice * multiplier
                    
                    -- print(string.format("--> Historical Pricing: Field %d price adjusted to $%d (multiplier: %.2f)", self.id or 0, self.price, multiplier))
                end
            end
        end
        
        print("--> Historical Pricing: Farmland price hook installed successfully")
    else
        print("--> Historical Pricing: WARNING - Could not install farmland price hook")
        print("  Farmland exists: " .. tostring(Farmland ~= nil))
        if Farmland then
            print("  Farmland.updatePrice exists: " .. tostring(Farmland.updatePrice ~= nil))
        end
    end
end

-- Force sync the year to a specific client (Logic used during join)
function HistoricalPricing:syncYearToClient(connection)
    if self.customYear and HistoricalPricingYearEvent then
        print("--> Historical Pricing: Syncing initial year " .. tostring(self.customYear) .. " to client (Connection: " .. tostring(connection) .. ")")
        connection:sendEvent(HistoricalPricingYearEvent.new(self.customYear))
    end
end

function HistoricalPricing:consoleCommandSyncLogin()
    if not self.isClient then -- Server side
         print("--> Historical Pricing: Simulating login sync for all connected users...")
         
         if g_currentMission and g_currentMission.userManager then
            local users = g_currentMission.userManager:getUsers()
            for _, user in pairs(users) do
                local connection = user:getConnection()
                if connection then
                    self:syncYearToClient(connection)
                end
            end
            return "Historical Pricing: Simulated login sync for all users."
         else
            return "Error: User manager not found."
         end
    else
        return "Error: Command must be run on server."
    end
end

-- Hook to synchronize year with new clients when they join
FSBaseMission.sendInitialClientState = Utils.appendedFunction(FSBaseMission.sendInitialClientState,
    function(self, connection, user, farm)
        if HistoricalPricing then
            HistoricalPricing:syncYearToClient(connection)
        end
    end
)

-- Console Commands for Testing
function HistoricalPricing:consoleCommandSetYear(year)
    year = tonumber(year)
    if year == nil then
        return "Usage: hpSetYear <year>"
    end

    if not self.isClient then -- Server side
        print(string.format("--> Historical Pricing: Console command forcing year to %d", year))
        self.customYear = year
        HistoricalPricing.writeSettings()
        self:updatePrices()
        
        -- Broadcast change
        if HistoricalPricingYearEvent then
            HistoricalPricingYearEvent.sendEvent(self.customYear)
        end
        return string.format("Historical Pricing: Year set to %d and broadcasted to clients", year)
    else -- Client side
        -- Send request to server
        if HistoricalPricingYearEvent then
            HistoricalPricingYearEvent.sendEvent(year)
        end
        return "Historical Pricing: Sent year change request to server"
    end
end

function HistoricalPricing:consoleCommandNextYear()
    if self.customYear then
        return self:consoleCommandSetYear(self.customYear + 1)
    else
        return "Error: Custom year not initialized"
    end
end

-- Draw year on HUD (below the month display)
function HistoricalPricing:draw()
    if not self.isInitialized or not self.customYear then
        return
    end
    
    -- Only draw when HUD is visible and we're in game
    if not g_currentMission.hud or not g_currentMission.hud:getIsVisible() then
        return
    end
    
    -- Position directly below the month display (AUG box)
    -- Centered under the month box
    local x = 0.812
    local y = 0.920  -- Tighter below the month box
    
    -- Get appropriate font size (scale with screen)
    local textSize = getCorrectTextSize(0.016)
    
    -- Set text properties
    setTextColor(1, 1, 1, 1)  -- White color
    setTextAlignment(RenderText.ALIGN_CENTER)
    setTextBold(false)
    
    -- Render the year
    renderText(x, y, textSize, tostring(self.customYear))
    
    -- Reset text alignment
    setTextAlignment(RenderText.ALIGN_LEFT)
end

-- Register the mod
addModEventListener(HistoricalPricing)

