-- Historical Pricing Settings GUI Integration

HistoricalPricingSettings = {}

-- Define Settings
HistoricalPricing.SETTINGS = {
    customYear = { default = 2025 },
    currentMonth = { default = 1 }
}

function HistoricalPricing.writeSettings()
    local key = "historicalPricing"
    local userSettingsFile = Utils.getFilename("modSettings/HistoricalPricing.xml", getUserProfileAppPath())
    
    local xmlFile = createXMLFile("settings", userSettingsFile, key)
    if xmlFile ~= 0 then
        -- Year
        local year = HistoricalPricing.customYear or HistoricalPricing.SETTINGS.customYear.default
        setXMLInt(xmlFile, key..".customYear#value", year)
        
        -- Month
        local month = HistoricalPricing.currentMonth 
        if not month and g_currentMission and g_currentMission.environment then
             month = g_currentMission.environment.currentMonth
        end
        month = month or HistoricalPricing.SETTINGS.currentMonth.default
        setXMLInt(xmlFile, key..".currentMonth#value", month)

        saveXMLFile(xmlFile)
        delete(xmlFile)
        print("--> Historical Pricing: Settings saved to " .. userSettingsFile)
    end
end

function HistoricalPricing.readSettings()
    local userSettingsFile = Utils.getFilename("modSettings/HistoricalPricing.xml", getUserProfileAppPath())
    
    if not fileExists(userSettingsFile) then
        print("--> Historical Pricing: Creating user settings file: "..userSettingsFile)
        HistoricalPricing.customYear = HistoricalPricing.SETTINGS.customYear.default
        HistoricalPricing.writeSettings()
        return
    end
    
    local xmlFile = loadXMLFile("historicalPricing", userSettingsFile)
    if xmlFile ~= 0 then
        local key = "historicalPricing"
        
        -- Read Year
        local year = getXMLInt(xmlFile, key..".customYear#value")
        if year then
            HistoricalPricing.customYear = year
        else
            HistoricalPricing.customYear = HistoricalPricing.SETTINGS.customYear.default
        end
        
        -- Read Month
        local month = getXMLInt(xmlFile, key..".currentMonth#value")
        if month then
            HistoricalPricing.currentMonth = month
        end
        
        delete(xmlFile)
        print("--> Historical Pricing: Settings loaded. Year: " .. tostring(HistoricalPricing.customYear))
    end
end

-- Year lists will be populated from pricing data
HistoricalPricingSettings.yearTexts = {}
HistoricalPricingSettings.yearValues = {}

function HistoricalPricingSettings:loadYearsFromData()
    -- Clear existing data
    self.yearTexts = {}
    self.yearValues = {}
    
    -- Get years from the pricing data
    if HistoricalPricing.pricingData then
        for year, _ in pairs(HistoricalPricing.pricingData) do
            table.insert(self.yearValues, year)
        end
        
        -- Sort years
        table.sort(self.yearValues)
        
        -- Create text labels
        for _, year in ipairs(self.yearValues) do
            table.insert(self.yearTexts, tostring(year))
        end
        
        print("--> Historical Pricing Settings: Loaded " .. #self.yearValues .. " years from pricing data")
    else
        print("--> Historical Pricing Settings: WARNING - No pricing data available")
    end
end

function HistoricalPricingSettings:onFrameOpen()
    if self.historicalPricingSettingsDone then
        return
    end

    HistoricalPricingSettings.cachedFrame = self
    
    -- Load years from pricing data
    HistoricalPricingSettings:loadYearsFromData()
    
    -- Add settings section
    HistoricalPricingSettings:addTitle(self)
    
    -- Custom Year Setting
    HistoricalPricing.yearSetting = HistoricalPricingSettings:addMultiTextOption(
        self, 
        "onYearChanged", 
        HistoricalPricingSettings.yearTexts, 
        g_i18n:getText("setting_historicalPricingYear"), 
        g_i18n:getText("setting_historicalPricingYear_tooltip")
    )

    self.generalSettingsLayout:invalidateLayout()
    self:updateAlternatingElements(self.generalSettingsLayout)
    self:updateGeneralSettings(self.generalSettingsLayout)

    self.historicalPricingSettingsDone = true
    HistoricalPricingSettings:updateSettings()
end

function HistoricalPricingSettings:addTitle(inGameMenuSettingsFrame)
    local textElement = TextElement.new()
    local textElementProfile = g_gui:getProfile("fs25_settingsSectionHeader")
    textElement.name = "sectionHeader"
    textElement:loadProfile(textElementProfile, true)
    textElement:setText(g_i18n:getText("historicalPricing_sectionHeader"))
    inGameMenuSettingsFrame.generalSettingsLayout:addElement(textElement)
    textElement:onGuiSetupFinished()
    textElement.focusId = FocusManager:serveAutoFocusId()
end

function HistoricalPricingSettings:addMultiTextOption(inGameMenuSettingsFrame, onClickCallback, texts, title, tooltip)
    local bitMap = BitmapElement.new()
    local bitMapProfile = g_gui:getProfile("fs25_multiTextOptionContainer")
    bitMap:loadProfile(bitMapProfile, true)
    bitMap.focusId = FocusManager:serveAutoFocusId()

    local multiTextOption = MultiTextOptionElement.new()
    local multiTextOptionProfile = g_gui:getProfile("fs25_settingsMultiTextOption")
    multiTextOption:loadProfile(multiTextOptionProfile, true)
    multiTextOption.target = HistoricalPricingSettings
    HistoricalPricingSettings.name = inGameMenuSettingsFrame.name
    multiTextOption:setCallback("onClickCallback", onClickCallback)
    multiTextOption:setTexts(texts)
    multiTextOption.focusId = FocusManager:serveAutoFocusId()

    local multiTextOptionTitle = TextElement.new()
    local multiTextOptionTitleProfile = g_gui:getProfile("fs25_settingsMultiTextOptionTitle")
    multiTextOptionTitle:loadProfile(multiTextOptionTitleProfile, true)
    multiTextOptionTitle:setText(title)
    multiTextOptionTitle.focusId = FocusManager:serveAutoFocusId()

    local multiTextOptionTooltip = TextElement.new()
    local multiTextOptionTooltipProfile = g_gui:getProfile("fs25_multiTextOptionTooltip")
    multiTextOptionTooltip.name = "ignore"
    multiTextOptionTooltip:loadProfile(multiTextOptionTooltipProfile, true)
    multiTextOptionTooltip:setText(tooltip)
    multiTextOptionTooltip.focusId = FocusManager:serveAutoFocusId()

    multiTextOption:addElement(multiTextOptionTooltip)
    bitMap:addElement(multiTextOption)
    bitMap:addElement(multiTextOptionTitle)

    FocusManager:loadElementFromCustomValues(bitMap, nil, nil, false, false)
    FocusManager:loadElementFromCustomValues(multiTextOption, nil, nil, false, false)
    FocusManager:loadElementFromCustomValues(multiTextOptionTitle, nil, nil, false, false)

    multiTextOption:onGuiSetupFinished()
    multiTextOptionTitle:onGuiSetupFinished()
    multiTextOptionTooltip:onGuiSetupFinished()

    inGameMenuSettingsFrame.generalSettingsLayout:addElement(bitMap)
    bitMap:onGuiSetupFinished()
    
    return multiTextOption
end

function HistoricalPricingSettings:updateSettings()
    -- Update Year setting
    if HistoricalPricing.yearSetting ~= nil and HistoricalPricing.customYear ~= nil then
        for index, value in pairs(HistoricalPricingSettings.yearValues) do
            if HistoricalPricing.customYear == value then
                HistoricalPricing.yearSetting:setState(index)
                break
            end
        end
    end
end

-- Callback function for year change
function HistoricalPricingSettings:onYearChanged(state)
    -- Check if player has permission to change settings (admin/server only)
    local canChange = false
    
    if g_currentMission:getIsServer() then
        -- Server can always change
        canChange = true
    elseif g_currentMission.isMasterUser then
        -- Master user (admin) can change
        canChange = true
    end
    
    if not canChange then
        print("--> Historical Pricing: Only server/admin can change the year setting")
        -- Reset to current year
        HistoricalPricingSettings:updateSettings()
        return
    end
    
    local newYear = HistoricalPricingSettings.yearValues[state]
    HistoricalPricingSettings.cachedFrame:playSample(GuiSoundPlayer.SOUND_SAMPLES.CLICK)
    print("--> Historical Pricing: Year setting changed to " .. newYear)
    
    -- If we are the server, save and broadcast
    if g_currentMission:getIsServer() then
    	print("--> Historical Pricing: Server applying year change")
        HistoricalPricing.customYear = newYear
        HistoricalPricing.writeSettings()
        HistoricalPricing:updatePrices()
        
        if HistoricalPricingYearEvent then
            HistoricalPricingYearEvent.sendEvent(newYear)
        end
    else
        -- If we are a client admin, send request to server
        print("--> Historical Pricing: Client admin requesting year change from server")
        if HistoricalPricingYearEvent then
            HistoricalPricingYearEvent.sendEvent(newYear)
        end
        
        -- We don't change local year yet - we wait for server to send it back via event
        -- ensure smooth UI feedback though:
        HistoricalPricing.customYear = newYear
        HistoricalPricing:updatePrices()
    end
end

-- Hook into the settings frame
InGameMenuSettingsFrame.onFrameOpen = Utils.appendedFunction(InGameMenuSettingsFrame.onFrameOpen, function(self)
    -- First time: add controls
    if not self.historicalPricingSettingsDone then
        HistoricalPricingSettings.cachedFrame = self
        
        -- Load years from pricing data
        HistoricalPricingSettings:loadYearsFromData()
        
        -- Add settings section
        HistoricalPricingSettings:addTitle(self)
        
        -- Custom Year Setting
        HistoricalPricing.yearSetting = HistoricalPricingSettings:addMultiTextOption(
            self, 
            "onYearChanged", 
            HistoricalPricingSettings.yearTexts, 
            g_i18n:getText("setting_historicalPricingYear"), 
            g_i18n:getText("setting_historicalPricingYear_tooltip")
        )

        self.generalSettingsLayout:invalidateLayout()
        self:updateAlternatingElements(self.generalSettingsLayout)
        self:updateGeneralSettings(self.generalSettingsLayout)

        self.historicalPricingSettingsDone = true
    end
    
    -- Always update the state when frame opens (like LumberJack does)
    HistoricalPricingSettings:updateSettings()
    
    -- Enable/disable based on permissions
    if HistoricalPricing.yearSetting then
        local isAdmin = g_currentMission:getIsServer() or g_currentMission.isMasterUser
        HistoricalPricing.yearSetting:setDisabled(not isAdmin)
    end
end)

-- Allow keyboard/controller navigation of menu options (like LumberJack)
FocusManager.setGui = Utils.appendedFunction(FocusManager.setGui, function(_, gui)
    if gui == "ingameMenuSettings" then
        -- Re-register our control with the focus manager
        if HistoricalPricing.yearSetting then
            if not HistoricalPricing.yearSetting.focusId or not FocusManager.currentFocusData.idToElementMapping[HistoricalPricing.yearSetting.focusId] then
                FocusManager:loadElementFromCustomValues(HistoricalPricing.yearSetting, nil, nil, false, false)
            end
        end
        -- Invalidate the layout so the up/down connections are analyzed again by the focus manager
        local settingsPage = g_gui.screenControllers[InGameMenu].pageSettings
        if settingsPage and settingsPage.generalSettingsLayout then
            settingsPage.generalSettingsLayout:invalidateLayout()
        end
    end
end)
