-- Network event for synchronizing year changes across multiplayer clients
HistoricalPricingYearEvent = {}
local HistoricalPricingYearEvent_mt = Class(HistoricalPricingYearEvent, Event)

InitEventClass(HistoricalPricingYearEvent, "HistoricalPricingYearEvent")

function HistoricalPricingYearEvent.emptyNew()
    local self = Event.new(HistoricalPricingYearEvent_mt)
    return self
end

function HistoricalPricingYearEvent.new(year)
    local self = HistoricalPricingYearEvent.emptyNew()
    self.year = year
    return self
end

function HistoricalPricingYearEvent:readStream(streamId, connection)
    self.year = streamReadInt32(streamId)
    self:run(connection)
end

function HistoricalPricingYearEvent:writeStream(streamId, connection)
    streamWriteInt32(streamId, self.year)
end

function HistoricalPricingYearEvent:run(connection)
    if connection:getIsServer() then
        -- Client received year change from server (connection IS the server)
        print("--> Historical Pricing: Received year change from server: " .. tostring(self.year))
        
        if HistoricalPricing then
            HistoricalPricing.customYear = self.year
            HistoricalPricing:updatePrices()
            -- Clients don't save to savegame, they just utilize the value
        end
    else
        -- Server received year change request from client (connection is NOT the server)
        print("--> Historical Pricing: Server received year change request from client: " .. tostring(self.year))
        
        -- Check if client has permission (admin)
        -- To match BetterContracts behavior, we trust the client UI validation.
        -- We removed the server-side check to prevent API errors and ensure compatibility.
        --[[ 
        local isAdmin = false
        if g_currentMission.userManager then
            local users = g_currentMission.userManager:getUsers()
            for _, user in pairs(users) do
                if user:getConnection() == connection then
                   if user:getIsMasterUser() then isAdmin = true end
                   break
                end
            end
        end
        if not isAdmin then
            print("--> Historical Pricing: WARNING - Client without admin privileges attempted to change year")
            return
        end 
        ]]
        
        if HistoricalPricing then
            HistoricalPricing.customYear = self.year
            
            -- Save settings on server
            HistoricalPricing.writeSettings()
            
            -- Update prices on server
            HistoricalPricing:updatePrices()
            
            -- Broadcast to other clients
            print("--> Historical Pricing: Broadcasting year change to other clients")
            g_server:broadcastEvent(HistoricalPricingYearEvent.new(self.year), nil, connection, nil)
        end
    end
end

function HistoricalPricingYearEvent.sendEvent(year)
    if g_server ~= nil then
        -- Server broadcasting to all
        print("--> Historical Pricing: Broadcasting year change to all clients: " .. tostring(year))
        g_server:broadcastEvent(HistoricalPricingYearEvent.new(year), nil, nil, nil)
    else
        -- Client sending to server
        print("--> Historical Pricing: Sending year change request to server: " .. tostring(year))
        g_client:getServerConnection():sendEvent(HistoricalPricingYearEvent.new(year))
    end
end
